import Image from "next/image";
import DmcBanner from "@/app/components/dmc/dmc-banner";
import Breadcrumb from "@/app/components/common/Breadcrumb";
import ExpandableText from "@/app/components/common/ExpandableText";
import Link from "next/link";
import { fetchDmcPageDetails } from "@/app/services/dmcServices";
import { getCanonical } from "@/app/lib/getCanonical";

// Define interfaces for TypeScript
interface Service {
  title: string;
  description: string;
  icon: string;
  color: string;
}

interface Strength {
  title: string;
  description: string;
  icon: string;
  color: string;
}

export async function generateMetadata() {
  const data = await fetchDmcPageDetails("dmc");

  const meta = data?.meta || {};
  const canonical = await getCanonical("/indian-dmc");
  const currentUrl = canonical;

  // Extract the meta_details from API
  const metaDetails = meta.meta_details || "";

  return {
    title: meta?.meta_title || "cholan tours",
    description: meta?.meta_description || "cholan tours",
    keywords: meta.meta_keywords || "",
    alternates: { canonical },

    openGraph: {
      title: meta?.meta_title || "cholan tours",
      url: currentUrl,
      description: meta?.meta_description || "cholan tours",
    },

    twitter: {
      title: meta?.meta_title || "cholan tours",
      url: currentUrl,
      description: meta?.meta_description || "cholan tours",
    },

    // Dynamically inject the meta_details content into head
    // other: {
    //   // This will render the raw HTML from meta_details in the head section
    //   "meta-details": metaDetails,
    // },
  };
}

const staticBreadcrumbItems = [
  { label: "Home", href: "/" },
  { label: "Our Partners", isCurrent: true },
];

export default async function Home() {
  const pageData = await fetchDmcPageDetails("dmc");
  // Dynamic data for services
  const services: Service[] = [
    {
      title: "Customized Itineraries",
      description:
        "Tailor-made travel programs across India, Nepal, Bhutan, and Sri Lanka.",
      icon: "bi-map",
      color: "text-primary",
    },
    {
      title: "Hotel & Transport",
      description:
        "Hotel contracting and an in-house fleet of modern vehicles.",
      icon: "bi-house-door",
      color: "text-warning",
    },
    {
      title: "Special Interest Tours",
      description:
        "Wellness, Culture, Heritage, Wildlife, Adventure, and more.",
      icon: "bi-tree",
      color: "text-success",
    },
    {
      title: "MICE Services",
      description: "Meetings, Incentives, Conferences, and Exhibitions.",
      icon: "bi-people",
      color: "text-info",
    },
    {
      title: "24/7 Support",
      description: "Multilingual professionals for round-the-clock assistance.",
      icon: "bi-headset",
      color: "text-danger",
    },
  ];

  // Dynamic data for strengths
  const strengths: Strength[] = [
    {
      title: "Pan-India Presence",
      description: "Over 15 offices across major tourist hubs.",
      icon: "bi-geo-alt-fill",
      color: "text-primary",
    },
    {
      title: "Dedicated Team",
      description: "400+ experienced professionals for personalized service.",
      icon: "bi-people-fill",
      color: "text-success",
    },
    {
      title: "Own Fleet",
      description: "Modern, GPS-enabled vehicles for safety and comfort.",
      icon: "bi-truck",
      color: "text-warning",
    },
    {
      title: "Sustainability Focus",
      description: "100% solar-powered, paperless offices.",
      icon: "bi-sun",
      color: "text-danger",
    },
    {
      title: "Technology Integration",
      description: "Real-time booking and tracking software.",
      icon: "bi-laptop",
      color: "text-info",
    },
  ];

  const staticBreadcrumbItems = [
    { label: "Home", href: "/" },
    { label: "Indian DMC", isCurrent: true },
  ];

  function renderBreadcrumbHTML(items: any) {
    if (!Array.isArray(items)) return "";
    return `
      <nav class="breadcrumb text-sm mb-4">
        ${items
          .map(
            (item, idx) =>
              `<a href="${item.href}" class="text-blue-600 hover:underline">${
                item.label
              }</a>${idx < items.length - 1 ? " / " : ""}`
          )
          .join("")}
      </nav>
    `;
  }

  return (
    <div>
      {/* Hero Section */}
      {pageData?.banner_image && (
        // <DmcBanner
        //   data={[
        //     {
        //       title: pageData?.title || "",
        //       subtitle: pageData?.short_description || "",
        //       imageUrl: pageData?.banner_image,
        //       imageUrlAlt:
        //         pageData?.banner_image_alt || pageData?.title || "Banner Image",
        //     },
        //   ]}
        // />

        <DmcBanner
          title={pageData?.title}
          subtitle={pageData?.short_description}
          imageUrl={pageData?.banner_image}
        />
      )}

      <section className="pt-4">
        <div className="container">
          <Breadcrumb items={staticBreadcrumbItems} />
        </div>
      </section>

      <div
        className="cms-page-content"
        dangerouslySetInnerHTML={{
          __html:
            pageData?.description
              ?.replaceAll(
                "{staticBreadcrumbItems}",
                JSON.stringify(staticBreadcrumbItems)
              )
              ?.replaceAll(
                '<breadcrumb items="{staticBreadcrumbItems}"></breadcrumb>',
                `<div class="breadcrumb-container">${renderBreadcrumbHTML(
                  staticBreadcrumbItems
                )}</div>`
              ) || "",
        }}
      ></div>

      <div className="row justify-content-center gap-4 mb-5">
        <div className="col-lg-10 text-center">
          <div className="btm-btn">
            <Link href={`/contact-us`} className="btn blue-btn">
              Contact Us
              <span>
                <Image
                  width={23}
                  height={23}
                  sizes="100vw"
                  src="/images/button-arrow.png"
                  alt="/images/no-img.webp"
                />
              </span>
            </Link>
          </div>
        </div>
      </div>
    </div>
  );
}
