<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use Illuminate\Support\Str;

class CategoryController extends Controller {

    public function index(Request $r){ 
        if ($r->exists('id')) {
            return response()->json(['category'=> Category::where('is_deleted','!=',1)->where('id',$r->id)->first()]);
        } else {
            $categories = Category::where('is_deleted','!=',1)
            ->withCount('packages')
            ->orderBy('id','desc')
            ->paginate(25); 
            return view('admin.categories.index', compact('categories')); 
        }
    }

    public function create(){ 
        //echo 'hvgjhghjdaj';
        return view('admin.categories.create'); 
    }

    public function store(Request $r){
        $validated = $r->validate([
            'name'        => 'required|max:150',
            'description' => 'required|max:500',
            'slug'        => 'required|max:150|unique:categories,slug',
            'title'       => 'required|max:150',
            'sub_title'   => 'required|max:150',
            'banner_image'=> 'required|image|mimes:webp'
        ]);
        $path = '';
        if ($r->hasFile('banner_image')) {
            $path = $r->file('banner_image')->store('category', 's3');
        }
        $category = Category::create([
            'name'        => $validated['name'],
            'slug'        => Str::slug($validated['name']),
            'title'       => $validated['title'],
            'sub_title'   => $validated['sub_title'],
            'banner_image'=> $path,
            'description' => $validated['description']
        ]);

        if ($r->ajax()) {
            return response()->json([
                'success'  => true,
                'message'  => 'Category created successfully',
                'category' => $category
            ]);
        }
        // Fallback for non-AJAX
        return redirect()
        ->route('admin.categories.index')
        ->with('success', 'Category updated successfully');
    }


    public function edit(Category $category){
        if (request()->ajax()) {
            return response()->json($category);
        }
    }


    public function update(Request $r, Category $category){

        if (!$r->exists('status')) {
            $r->validate([
                'name'        => 'required|max:150',
                'title'       => 'required|max:150',
                'sub_title'   => 'required|max:150',
                'description' => 'required|max:500',
                'banner_image'=> 'nullable|image|mimes:webp',
                'slug'        => 'required|max:256|unique:categories,slug,' . $category->id
            ]);
            $path = $category->banner_image ?? '';
            if ($r->hasFile('banner_image')) {
                // delete old file if it exists in S3
                if ($category->banner_image && \Storage::disk('s3')->exists($category->banner_image)) {
                    \Storage::disk('s3')->delete($category->banner_image);
                }

                $file = $r->file('banner_image');
                $filename = $file->getClientOriginalName(); // keep original filename

                // store in S3 under category folder
                $path = $file->storeAs('category', $filename, 's3');

                $category->banner_image = $path; // save path in model
            }


            $category->update([
                'name'        => $r->name,
                'title'       => $r->title,
                'sub_title'   => $r->sub_title,
                'banner_image'=> $path,
                'description' => $r->description,
                'slug'        => $r->slug
            ]);
        } else {
            $category->is_active = $r->status;
            $category->save();
        }

        // If AJAX request, return JSON response
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Category updated successfully',
                'data'    => $category
            ]);
        }

        // Fallback for non-AJAX
        return redirect()
        ->route('admin.categories.index')
        ->with('success', 'Category updated successfully');
    }


    public function destroy(Category $category){ 
        $category->is_deleted = 1; 
        $category->save();
        if ($category->banner_image && \Storage::disk('s3')->exists($category->banner_image)) {
            \Storage::disk('s3')->delete($category->banner_image);
        }
        return response()->json(['success'=>true]);
    }

    public function slugDuplicateCheck(Request $r){ 
        return response()->json([
            'exists' => Category::where('slug', $r->slug)
            ->where('id', '!=', $r->id)
            ->where('is_deleted', '!=', 1)
            ->exists()
        ]);

    }
}
