<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ContinentCountry;
use App\Models\Continent;
use App\Models\Country;
use App\Models\Location;
use App\Models\CountryDetails;
use App\Models\CountryFaq;
use App\Models\CountryMetaData;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class CountryController extends Controller{

    public function index(Request $r){
        if ($r->exists('continent_id') && !$r->exists('faqs')) {
            return response()->json(['countries'=>Country::where('continent_id',$r->continent_id)->get()]);
        } elseif(!$r->exists('continent_id') && $r->exists('faqs')){
            $faqs = CountryFaq::where('country_id',$r->id)->get();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'faqs'    => $faqs
                ]);
            }
        }
    }

    // public function getCountries(Request $request){
    //     if ($request->ajax()) {
    //         return response()->json([
    //             'html' => view('admin.locations.countries', compact('countries'))->render(),
    //             'pagination' => (string) $countries->links(),
    //         ]);
    //     }
    // }

  

    public function store(Request $r){
        $r->validate([
            'continent_id' => 'required|exists:continents,id',
            'name' => 'required|string|max:255'
        ]);
        $slug = Str::slug($r->name.'-tour-packages');

        if (Country::where('slug', $slug)->exists()) {
            throw ValidationException::withMessages([
                'name' => 'Country already exists.',
            ]);
        }
        $country = Country::create([
            'continent_id' => $r->continent_id,
            'slug' => $slug,
            'name' => $r->name,
            'code' => $r->code
        ]);

        return response()->json($country->load('continent'));
    }

    public function show(Country $country){
        $country->load('continent','details');
        return response()->json($country);
    }

    public function showMeta(Country $country){
        $country->load('meta');
        return response()->json($country);
    }

    public function update(Request $r, country $country){
        if(!$r->exists('meta_setting')){
            $r->validate([
                'banner_image' => 'nullable|image|mimes:webp'
            ]);
            $path = $country->details->banner_image ?? '';
            if ($r->hasFile('banner_image')) {
                $img = $r->file('banner_image');

                if ($img->isValid()) {
                    // delete old file if it exists
                    if ($country->details && $country->details->banner_image && \Storage::disk('s3')->exists($country->details->banner_image)) {
                        \Storage::disk('s3')->delete($country->details->banner_image);
                    }

                    // keep original filename
                    $filename = $img->getClientOriginalName();

                    // store with original filename in S3 under "country" folder
                    $path = $img->storeAs('country', $filename, 's3');

                    // save path wherever needed
                    //$country->details->banner_image = $path;
                    //$country->details->save();
                }
            }

            if ($country->details) {
                $country->details->update([
                    'title' => $r->title,
                    'sub_title' => $r->sub_title,
                    'banner_image' => $path,
                    'banner_image_alt'=>$r->banner_image_alt,
                    'about' => $r->about
                ]);
            } else {
                CountryDetails::create([
                    'country_id' => $country->id,
                    'title' => $r->title,
                    'sub_title' => $r->sub_title,
                    'banner_image' => $path,
                    'banner_image_alt'=>$r->banner_image_alt,
                    'about' => $r->about
                ]);
            }
        } else {
            $r->validate([
                'meta_title'       => 'nullable|string',
                'meta_description' => 'nullable|string',
                'meta_keywords'    => 'nullable|string',
                'h1_heading'       => 'nullable|string',
                'meta_details'     => 'nullable|string',
            ]);
            if ($country->meta) {
                $country->meta->update([
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            } else {
                CountryMetaData::create([
                    'country_id' => $country->id,
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            }
        }
        return redirect()->route('admin.locations.index')->with('success', 'Country updated successfully');
    }

    public function updateFaq(Request $r, Country $country){
        $country->faqs()->delete();
        if ($r->has('faqs')) {
            foreach ($r->faqs as $obj) {
                CountryFaq::create([
                    'country_id'=>$country->id,
                    'question' => $obj['question'],
                    'answer'   => $obj['answer'] ?? null,
                ]);
            }
        }
        return redirect()->route('admin.locations.index')->with('success', 'Country faq updated successfully');
    }

    public function masterCountries(Request $r){
        return response()->json(['countries'=>ContinentCountry::where('continent_code',$r->code)->get()]);
    }
    
}
