<?php
namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

use App\Models\Package;
use App\Models\PackageDetail;
use App\Models\PackageImage;
use App\Models\PackageItinerary;
use App\Models\PackageFaq;
use App\Models\PackageMetaData;
use App\Models\Category;
use App\Models\Continent;
use App\Models\Country;
use App\Models\Location;
use App\Models\Facility;


class PackageController extends Controller {
    public function index(Request $r){
        if ($r->exists('id') && !$r->exists('faqs')) {
            $package = Package::where('id',$r->id)->with('category','details','location.country.continent','images','itineraries','faqs','source_location.country.continent')->first();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'package'    => $package
                ]);
            }
        } elseif($r->exists('id') && $r->exists('faqs')){
            $faqs = PackageFaq::where('package_id',$r->id)->get();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'faqs'    => $faqs
                ]);
            }
        } else {
            $packages = Package::where('is_deleted','!=',1)
            ->with('category','details','location.country.continent')
            ->orderBy(list_config()['order_by'], list_config()['direction'])
            ->paginate(25);
            return view('admin.packages.index', compact('packages'));
        }
    }

    public function create(){
        $categories = Category::where('is_active',1)->where('is_deleted',0)->orderBy('id','desc')->get();
        $continents = Continent::all();
        $facilities = Facility::where('is_active',1)->get();
        return view('admin.packages.create', compact('categories','continents','facilities'));
    }

    public function store(Request $r){
        $r->validate([
            'title'              => 'required|string|max:255',
            'type'               => 'required',
            'category_id'        => 'required|exists:categories,id',
            'start_date'         => 'nullable|date|after_or_equal:today',
            'end_date'           => 'nullable|date|after_or_equal:start_date',
            'duration_days'      => 'required|integer|min:1',
            'duration_nights'    => 'required|integer|min:0',
            'short_description'  => 'required|string',
            'continent_id'       => 'required|exists:continents,id',
            'country_id'         => 'required|exists:countries,id',
            'location_id'        => 'required|exists:locations,id',
            'source_continent_id'=> 'required|exists:continents,id',
            'source_country_id'  => 'required|exists:countries,id',
            'source_location_id' => 'required|exists:locations,id',
            'itinerary_overview' => 'required|string',
            //'includes'           => 'required|string',
            //'excludes'           => 'required|string',
            'tour_highlights'    => 'required|string',
            'route_details'      => 'required|string',
            'primary_image'      => 'required|image|mimes:webp',
            'images.*'           => 'required|image|mimes:webp',
        ]);

        $pkgData = $r->only([
            'title',
            'type',
            'category_id',
            //'price',
            'short_description',
            'long_description',
            'continent_id',
            'country_id',
            'location_id',
            'source_continent_id',
            'source_country_id',
            'source_location_id',
            'is_featured',
            'is_top_trending',
            'is_customized'
        ]);
        $pkgData['slug'] = Str::slug($r->title).'-'.$r->duration_days.'-days-'.$r->duration_nights.'-nights';
        // $pkgData['slug'] = 
        //     Str::slug(getPackageTypeName($r->type)).'-'.
        //     Str::slug(getCategoryName($r->category_id)).'-'.
        //     Str::slug(getLocationName($r->source_location_id)).'-to-'.
        //     Str::slug(getLocationName($r->location_id)).'-'.
        //     Str::slug($r->title).'-'.$r->duration_days.'-days-'.$r->duration_nights.'-nights';
        
        if ($r->hasFile('primary_image')) {
            $file     = $r->file('primary_image');
            $filename = $file->getClientOriginalName(); // original file name with extension
            $path     = $file->storeAs('packages', $filename, 's3');

            $pkgData['primary_image'] = $path !=0 ? $path : '';
        }
        $pkgData['primary_image_alt'] = $r->primary_image_alt;
        $package = Package::create($pkgData);
        PackageDetail::create([
            'package_id'=>$package->id,
            'start_date'=>$r->start_date ?? null,
            'end_date'=>$r->end_date ?? null,
            'duration_days'=>$r->duration_days,
            'duration_nights'=>$r->duration_nights,
            'itinerary_overview'=>$r->itinerary_overview,
            'includes'=>$r->includes ?? null,
            'excludes'=>$r->excludes ?? null,
            'tour_highlights'=>$r->tour_highlights,
            'route_details' => $r->route_details,
            'facilities' => $r->facilities ?? []

        ]);
        if ($r->hasFile('images')) {
            foreach ($r->file('images') as $key => $img) {
                if ($img->isValid()) {
                    // keep original filename
                    $filename = $img->getClientOriginalName();

                    // store with original name in S3
                    $p = $img->storeAs('packages', $filename, 's3');

                    // save to DB
                    PackageImage::create([
                        'package_id' => $package->id,
                        'image_path' => $p !=0 ? $p : '',
                        'image_alt' => $r->images_alt[$key] ?? null,
                    ]);
                }
            }
        }


        if ($r->has('itineraries')) {
            foreach ($r->itineraries as $obj) {
                PackageItinerary::create(['package_id'=>$package->id,'title'=>$obj['title'],'details'=>$obj['details']]);
            }
        }

        return redirect()->route('admin.packages.index')->with('success','Package created');
    }

    public function edit(Package $package){
        $categories = Category::where('is_active',1)->where('is_deleted',0)->orderBy('id','desc')->get();
        $continents = Continent::all();
        $countries = Country::where('continent_id',$package->continent_id)->get();
        $locations = Location::where('country_id',$package->country_id)->get();
        $source_countries = Country::where('continent_id',$package->source_continent_id)->get();
        $source_locations = Location::where('country_id',$package->source_country_id)->get();
        $package->load('details','images','itineraries');
        $facilities = Facility::where('is_active',1)->get();
        return view('admin.packages.edit', compact('package','categories','continents','countries','locations','source_countries','source_locations','facilities'));
    }

    public function showMeta(Package $package){
        $package->load('meta');
        return response()->json($package);
    }

    public function update(Request $r, Package $package){
        if (!$r->exists('status') && !$r->exists('meta_setting')) {
            $r->validate([
                'title'              => 'required|string|max:255',
                'type'               => 'required',
                'category_id'        => 'required|exists:categories,id',
                'start_date'         => 'nullable|date|after_or_equal:today',
                'end_date'           => 'nullable|date|after_or_equal:start_date',
                'duration_days'      => 'required|integer|min:1',
                'duration_nights'    => 'required|integer|min:0',
                //'price'              => 'nullable|numeric|min:0',
                'short_description'  => 'required|string',
                'continent_id'       => 'required|exists:continents,id',
                'country_id'         => 'required|exists:countries,id',
                'location_id'        => 'required|exists:locations,id',
                'source_continent_id'=> 'required|exists:continents,id',
                'source_country_id'  => 'required|exists:countries,id',
                'source_location_id' => 'required|exists:locations,id',
                'itinerary_overview' => 'required|string',
                //'includes'           => 'required|string',
                //'excludes'           => 'required|string',
                'tour_highlights'    => 'required|string',
                'route_details'      => 'required|string',
                'primary_image'      => 'nullable|image|mimes:webp',
                'images'             => 'nullable',
                'images.*'           => 'nullable|image|mimes:webp',
            ]);


            $pkgData = $r->only([
                'title',
                'type',
                'category_id',
                //'price',
                'short_description',
                'long_description',
                'continent_id',
                'country_id',
                'location_id',
                'source_continent_id',
                'source_country_id',
                'source_location_id',
                'is_featured',
                'is_top_trending',
                'is_customized'
            ]);

            if(!isset($pkgData['is_featured'])){
                $pkgData['is_featured'] = 0;
            }
            if(!isset($pkgData['is_top_trending'])){
                $pkgData['is_top_trending'] = 0;
            }
            if(!isset($pkgData['is_customized'])){
                $pkgData['is_customized'] = 0;
            }

            $pkgData['slug'] = Str::slug($r->title).'-'.$r->duration_days.'-days-'.$r->duration_nights.'-nights';
            // $pkgData['slug'] =
            //     Str::slug(getPackageTypeName($r->type)).'-'.
            //     Str::slug(getCategoryName($r->category_id)).'-'.
            //     Str::slug(getLocationName($r->source_location_id)).'-to-'.
            //     Str::slug(getLocationName($r->location_id)).'-'.
            //     Str::slug($r->title).'-'.$r->duration_days.'-days-'.$r->duration_nights.'-nights';

            // Handle primary image update
            if ($r->hasFile('primary_image')) {
                // delete old file if exists
                try{
                    if ($package->primary_image && $package->primary_image != 0 && \Storage::disk('s3')->exists($package->primary_image)) {
                        \Storage::disk('s3')->delete($package->primary_image);
                    }

                } catch (\Exception $e) {
                    // ✅ Catch any unexpected error, but don’t break deletion
                    \Log::error('Unexpected S3 error: '.$e->getMessage());
                }
                // get original filename
                $filename = $r->file('primary_image')->getClientOriginalName();

                // store with original filename in S3
                $path = $r->file('primary_image')->storeAs('packages', $filename, 's3');

                $pkgData['primary_image'] = $path !=0 ? $path : '';
            }
            $pkgData['primary_image_alt'] = $r->primary_image_alt;

            // Update package
            $package->update($pkgData);
            // Update package details
            if ($package->details) {

                $package->details->update([
                    'start_date'=>$r->start_date ?? null,
                    'end_date'=>$r->end_date ?? null,
                    'duration_days' => $r->duration_days,
                    'duration_nights' => $r->duration_nights,
                    'itinerary_overview' => $r->itinerary_overview,
                    'includes' => $r->includes ?? null,
                    'excludes' => $r->excludes ?? null,
                    'tour_highlights' => $r->tour_highlights,
                    'route_details' => $r->route_details,
                    'facilities' => $r->facilities ?? []
                ]);
            } else {
                PackageDetail::create([
                    'package_id' => $package->id,
                    'start_date'=>$r->start_date ?? null,
                    'end_date'=>$r->end_date ?? null,
                    'duration_days' => $r->duration_days,
                    'duration_nights' => $r->duration_nights,
                    'itinerary_overview' => $r->itinerary_overview,
                    'includes' => $r->includes ?? null,
                    'excludes' => $r->excludes ?? null,
                    'tour_highlights' => $r->tour_highlights,
                    'route_details' => $r->route_details,
                    'facilities' => $r->facilities ?? []
                ]);
            }

            // Handle gallery images (append new)
            // ✅ 1. If new images are uploaded
            if ($r->hasFile('images')) {
                foreach ($r->file('images') as $key => $img) {
                    if ($img->isValid()) {
                        $filename = $img->getClientOriginalName();

                        // store in S3
                        $p = $img->storeAs('packages', $filename, 's3');

                        PackageImage::create([
                            'package_id' => $package->id,
                            'image_path' => $p,
                            'image_alt'  => $r->images_alt[$key] ?? null,
                        ]);
                    }
                }
            }

            // ✅ 2. Update alt text for existing images
            if ($r->has('existing_images_alt')) {
                foreach ($r->existing_images_alt as $id => $alt) {
                    PackageImage::where('id', $id)->update([
                        'image_alt' => $alt,
                    ]);
                }
            }



            // Handle itineraries (simple: delete old & recreate)
            if ($r->has('itineraries')) {
                $package->itineraries()->delete();
                foreach ($r->itineraries as $obj) {
                    PackageItinerary::create([
                        'package_id' => $package->id,
                        'title' => $obj['title'],
                        'details' => $obj['details'],
                    ]);
                }
            }
        } 
        elseif($r->exists('meta_setting')) {
            
            $r->validate([
                'meta_title'       => 'nullable|string',
                'meta_description' => 'nullable|string',
                'meta_keywords'    => 'nullable|string',
                'h1_heading'       => 'nullable|string',
                'meta_details'     => 'nullable|string',
            ]);
            if ($package->meta) {
                //print_r($r->toArray());die;
                $package->meta->update([
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            } else {
                PackageMetaData::create([
                    'package_id' => $package->id,
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            }
        }
        elseif($r->exists('status')) {
            $package->is_active = $r->status;
            $package->save();
        }

        // If AJAX request, return JSON response
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Package status updated successfully',
                'data'    => $package
            ]);
        }

        return redirect()->route('admin.packages.index')->with('success', 'Package updated successfully');
    }

    public function updateFaq(Request $r, Package $package){
        $package->faqs()->delete();
        if ($r->has('faqs')) {
            foreach ($r->faqs as $obj) {
                PackageFaq::create([
                    'package_id'=>$package->id,
                    'question' => $obj['question'],
                    'answer'   => $obj['answer'] ?? null,
                ]);
            }
        }
        return redirect()->route('admin.packages.index')->with('success', 'Package faq updated successfully');
    }


    public function slugDuplicateCheck(Request $r){
        // $slug = Str::slug(getPackageTypeName($r->type)).'-'.
        // Str::slug(getCategoryName($r->category_id)).'-'.
        // Str::slug(getLocationName($r->source_location_id)).'-to-'.
        // Str::slug(getLocationName($r->location_id)).'-'.
        // Str::slug($r->title).'-'.$r->duration_days.'-days-'.$r->duration_nights.'-nights';
        $slug = Str::slug($r->title).'-'.$r->duration_days.'-days-'.$r->duration_nights.'-nights';
        $slugExists = Package::where('slug', $slug)
        ->when($r->exists('id')? $r->id : '', function ($query, $id) {
            return $query->where('id', '!=', $id);
        })
        ->where('is_deleted', '!=', 1)
        ->exists();
        //return response()->json(['exists' => getExecutedQuery($slugExists)]);
        return response()->json(['exists' => $slugExists]);
    }

    public function destroy(Package $package){
        $package->is_deleted = 1; 
        $package->save();
        if(Storage::disk('s3')->exists($package->primary_image)){
            Storage::disk('s3')->delete($package->primary_image);
        }
        foreach ($package->images as $img) { Storage::disk('s3')->delete($img->image_path); }
        return response()->json(['success'=>true]);
        // foreach ($package->images as $img) { Storage::disk('public')->delete($img->image_path); $img->delete(); }
        // if ($package->primary_image) Storage::disk('public')->delete($package->primary_image);
        // $package->delete();
        // return redirect()->route('admin.packages.index')->with('success','Deleted');
    }


    public function deleteImage(Request $r)
    {
        $img = PackageImage::find($r->id);

        if (!$img) {
            return response()->json(['success' => false, 'message' => 'Image not found'], 404);
        }

        try {

            // Try to delete from S3 safely
            if (Storage::disk('s3')->exists($img->image_path)) {
                Storage::disk('s3')->delete($img->image_path);
            }

        } catch (\Exception $e) {
            // ✅ Catch any unexpected error, but don’t break deletion
            \Log::error('Unexpected S3 error: '.$e->getMessage());
        }

        // ✅ Always remove from DB
        $img->delete();

        return response()->json(['success' => true]);
    }

}
