<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TourService;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;


class TourServiceController extends Controller{

    public function store(Request $r){
        $r->validate([
            'title' => 'required|string|max:255',
            'link' => 'required|string|max:255',
            'banner_image' => 'required|image|mimes:webp'
        ]);

        $obj = [
            'link' => $r->link,
            'title' => $r->title,
            'banner_image_alt' => $r->banner_image_alt
        ];
        session()->flash('active_tab', 'tservice');

        if ($r->hasFile('banner_image')) {
            $file = $r->file('banner_image');

            if ($file->isValid()) {
                // keep original filename
                $filename = $file->getClientOriginalName();

                // store in S3 under "tour_service/"
                $path = $file->storeAs('tour_service', $filename, 's3');

                $obj['banner_image'] = $path;
            }
        }


        $service = TourService::create($obj);

        return response()->json(['success'=>true]);
    }

    public function edit(TourService $tour_service){
        $tour_service = TourService::where('id',$tour_service->id)->first();
        return response()->json($tour_service);
    }

    public function update(Request $r, TourService $tour_service){

        if (!$r->exists('status')) {
            $r->validate([
                'title' => 'required|string|max:255',
                'link' => 'required|string|max:255',
                'banner_image' => 'nullable|image|mimes:webp'
            ]);
            $path = $tour_service->banner_image ?? '';
            if ($r->hasFile('banner_image')) {
                $file = $r->file('banner_image');

                if ($file->isValid()) {
                    // delete old file if exists
                    if ($tour_service->banner_image && \Storage::disk('s3')->exists($tour_service->banner_image)) {
                        \Storage::disk('s3')->delete($tour_service->banner_image);
                    }

                    // keep original filename
                    $filename = $file->getClientOriginalName();

                    // store in S3 under "banner/" folder
                    $path = $file->storeAs('banner', $filename, 's3');

                    // $tour_service->banner_image = $path;
                    // $tour_service->save();
                }
            }

            $obj = [
                'title'       => $r->title,
                'link'   => $r->link,
                'banner_image'=> $path,
                'banner_image_alt' => $r->banner_image_alt
            ];


            $tour_service->update($obj);
            session()->flash('active_tab', 'tservice');
        }  else {
            // Normal banner just update status
            $tour_service->is_active = $r->status ? 1 : 0;
            $tour_service->save();
        }
        
        // If AJAX request, return JSON response
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Service updated successfully',
                'data'    => $tour_service
            ]);
        }

        // Fallback for non-AJAX
        return redirect()
        ->route('admin.settings.index')
        ->with('success', 'Service updated successfully');
    }

    public function destroy(TourService $tour_service){
        $tour_service->is_deleted = 1; 
        $tour_service->save();
        if ($tour_service->banner_image && \Storage::disk('s3')->exists($tour_service->banner_image)) {
            \Storage::disk('s3')->delete($tour_service->banner_image);
        }
        return response()->json(['success'=>true]);
    }
}
