<?php
namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

use App\Models\Train;
use App\Models\TrainTour;
use App\Models\TrainTourDetails;
use App\Models\TrainTourCabin;
use App\Models\TrainTourItinerary;
use App\Models\TrainTourFaq;
use App\Models\Facility;
use App\Models\TrainTourMetaData;


class TrainTourController extends Controller {
    public function index(Request $r){
        if ($r->exists('id') && !$r->exists('faqs')) {
            $tour = TrainTour::where('id',$r->id)->with('train','details','itineraries','faqs')->first();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'tour'    => $tour
                ]);
            }
        } elseif($r->exists('id') && $r->exists('faqs')){
            $faqs = TrainTourFaq::where('train_tour_id',$r->id)->get();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'faqs'    => $faqs
                ]);
            }
        }
    }

    public function create(){
        $trains = Train::where('is_active',1)->where('is_deleted',0)->orderBy('id','desc')->get();
        $facilities = Facility::where('is_active',1)->get();
        return view('admin.train_tour.create', compact('trains','facilities'));
    }

    public function store(Request $r){
        $r->validate([
            'title'              => 'required|string|max:255',
            'train_id'           => 'required|exists:categories,id',
            'duration_days'      => 'required|integer|min:1',
            'duration_nights'    => 'required|integer|min:0',
            'description'        => 'required|string',
            'refund_cancellation_policy'=> 'required|string',
            'itinerary_overview' => 'required|string',
            //'includes'           => 'required|string',
            //'excludes'           => 'required|string',
            'route_details'      => 'required|string',
            'primary_image'      => 'required|mimes:webp',
            'map_image'          => 'required|mimes:jpg,jpeg,png,gif,webp'
        ]);

        $tourData = $r->only([
            'title',
            'train_id',
            'description',
            'refund_cancellation_policy'
        ]);
        $train = Train::where('id',$r->train_id)->first();
        $tourData['slug'] = Str::slug($r->title).'-'.$train->slug;
        
        if ($r->hasFile('primary_image')) {
            $file = $r->file('primary_image');

            if ($file->isValid()) {
                // keep original filename
                $filename = $file->getClientOriginalName();

                // store in S3 under "train_tours/"
                $path = $file->storeAs('train_tours', $filename, 's3');

                $tourData['primary_image'] = $path;
            }
        }

        $tour = TrainTour::create($tourData);

        $map_image = '';
        if ($r->hasFile('map_image')) {
            $file = $r->file('map_image');

            if ($file->isValid()) {
                // keep original filename
                $filename = $file->getClientOriginalName();

                // store in S3 under "train_tours_map/"
                $path = $file->storeAs('train_tours_map', $filename, 's3');

                $map_image = $path;
            }
        }

        TrainTourDetails::create([
            'train_tour_id'=>$tour->id,
            'duration_days'=>$r->duration_days,
            'duration_nights'=>$r->duration_nights,
            'itinerary_overview'=>$r->itinerary_overview,
            'includes'=>$r->includes ?? null,
            'excludes'=>$r->excludes ?? null,
            'route_details' => $r->route_details,
            'map_image' => $map_image,
            'facilities' => $r->facilities ?? []

        ]);

        if ($r->has('itineraries')) {
            foreach ($r->itineraries as $obj) {
                TrainTourItinerary::create(['train_tour_id'=>$tour->id,'title'=>$obj['title'],'details'=>$obj['details']]);
            }
        }

        return redirect()->route('admin.page-settings.luxury-train')
        ->with('active_tab','tour_list')
        ->with('success','Tour created');
    }

    public function edit(TrainTour $trainTour){
        $trains = Train::where('is_active',1)->where('is_deleted',0)->orderBy('id','desc')->get();
        $facilities = Facility::where('is_active',1)->get();
        $trainTour->load('details','itineraries');
        return view('admin.train_tour.edit', compact('trainTour','trains','facilities'));
    }

    public function showMeta(TrainTour $trainTour){
        $trainTour->load('meta');
        return response()->json($trainTour);
    }

    public function update(Request $r, TrainTour $trainTour){
        if (!$r->exists('status') && !$r->exists('meta_setting')) {
            $r->validate([
                'title'              => 'required|string|max:255',
                'train_id'           => 'required|exists:categories,id',
                'duration_days'      => 'required|integer|min:1',
                'duration_nights'    => 'required|integer|min:0',
                'description'        => 'required|string',
                'refund_cancellation_policy' => 'required|string',
                'itinerary_overview' => 'required|string',
                //'includes'           => 'required|string',
                //'excludes'           => 'required|string',
                'route_details'      => 'required|string',
                'primary_image'      => 'nullable|image|mimes:webp',
                'map_image'          => 'nullable|mimes:jpg,jpeg,png,gif,webp'
            ]);

            $tourData = $r->only([
                'title',
                'train_id',
                'description',
                'refund_cancellation_policy'
            ]);
            $train = Train::where('id',$r->train_id)->first();
            $tourData['slug'] = Str::slug($r->title).'-'.$train->slug;

            // Handle primary image update
            if ($r->hasFile('primary_image')) {
                $file = $r->file('primary_image');

                if ($file->isValid()) {
                    // delete old file if exists
                    if ($trainTour->primary_image && \Storage::disk('s3')->exists($trainTour->primary_image)) {
                        \Storage::disk('s3')->delete($trainTour->primary_image);
                    }

                    // keep original filename
                    $filename = $file->getClientOriginalName();

                    // store in S3 under "train_tours/"
                    $path = $file->storeAs('train_tours', $filename, 's3');

                    $tourData['primary_image'] = $path;
                }
            }


            // Update package
            $trainTour->update($tourData);
            
            $map_image = $trainTour->details->map_image;
            if ($r->hasFile('map_image')) {
                $file = $r->file('map_image');

                if ($file->isValid()) {
                    // delete old map image if exists
                    if ($trainTour->details->map_image && \Storage::disk('s3')->exists($trainTour->details->map_image)) {
                        \Storage::disk('s3')->delete($trainTour->details->map_image);
                    }

                    // keep original filename
                    $filename = $file->getClientOriginalName();

                    // store in S3 under "train_tours_map/"
                    $path = $file->storeAs('train_tours_map', $filename, 's3');

                    $map_image = $path;
                }
            }

            if ($trainTour->details) {

                $trainTour->details->update([
                    'duration_days' => $r->duration_days,
                    'duration_nights' => $r->duration_nights,
                    'itinerary_overview' => $r->itinerary_overview,
                    'includes' => $r->includes ?? null,
                    'excludes' => $r->excludes ?? null,
                    'route_details' => $r->route_details,
                    'map_image' => $map_image,
                    'facilities' => $r->facilities ?? []
                ]);
            } else {
                TrainTourDetails::create([
                    'train_tour_id' => $trainTour->id,
                    'duration_days' => $r->duration_days,
                    'duration_nights' => $r->duration_nights,
                    'itinerary_overview' => $r->itinerary_overview,
                    'includes' => $r->includes ?? null,
                    'excludes' => $r->excludes ?? null,
                    'route_details' => $r->route_details,
                    'map_image' => $map_image,
                    'facilities' => $r->facilities ?? []
                ]);
            }

            // Handle itineraries (simple: delete old & recreate)
            if ($r->has('itineraries')) {
                $trainTour->itineraries()->delete();
                foreach ($r->itineraries as $obj) {
                    TrainTourItinerary::create([
                        'train_tour_id' => $trainTour->id,
                        'title' => $obj['title'],
                        'details' => $obj['details'],
                    ]);
                }
            }
        }
        elseif($r->exists('meta_setting')) {
            
            $r->validate([
                'meta_title'       => 'nullable|string',
                'meta_description' => 'nullable|string',
                'meta_keywords'    => 'nullable|string',
                'h1_heading'       => 'nullable|string',
                'meta_details'     => 'nullable|string',
            ]);
            if ($trainTour->meta) {
                //print_r($r->toArray());die;
                $trainTour->meta->update([
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            } else {
                TrainTourMetaData::create([
                    'train_tour_id' => $trainTour->id,
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            }
        }
        elseif($r->exists('status')) {
            $trainTour->is_active = $r->status;
            $trainTour->save();
        }

        // If AJAX request, return JSON response
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Tour status updated successfully',
                'data'    => $trainTour
            ]);
        }

        return redirect()->route('admin.page-settings.luxury-train')
        ->with('active_tab','tour_list')
        ->with('success', 'Tour updated successfully');
    }

    public function updateFaq(Request $r, TrainTour $trainTour){
        $trainTour->faqs()->delete();
        if ($r->has('faqs')) {
            foreach ($r->faqs as $obj) {
                TrainTourFaq::create([
                    'train_tour_id'=>$trainTour->id,
                    'question' => $obj['question'],
                    'answer'   => $obj['answer'] ?? null,
                ]);
            }
        }
        return redirect()->route('admin.page-settings.luxury-train')
        ->with('active_tab','tour_list')
        ->with('success', 'Tour faq updated successfully');
    }


    public function slugDuplicateCheck(Request $r){
        $train = Train::where('id',$r->train_id)->first();
        $slug = Str::slug($r->title).'-'.$train->slug;

        $slugExists = TrainTour::where('slug', $slug)
        ->when($r->exists('id')? $r->id : '', function ($query, $id) {
            return $query->where('id', '!=', $id);
        })
        ->where('is_deleted', '!=', 1)
        ->exists();
        return response()->json(['exists' => $slugExists]);
    }

    public function destroy(TrainTour $trainTour){
        $trainTour->is_deleted = 1; 
        $trainTour->save();
        return response()->json(['success'=>true]);
    }

}
