<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Http\Resources\HotelResource;
use App\Models\Hotel;


class HotelController extends Controller{

    public function index(Request $r){

        $limit     = list_config()['limit']; // default 10 
        $orderBy   = list_config()['order_by']; // default order by id
        $direction = list_config()['direction']; // default desc

        $hotels = Hotel::where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->orderBy($orderBy, $direction)
            ->limit(4)
            ->get();

        $hotelIds = $hotels->pluck('id')->toArray();

        $topHotels = Hotel::where('is_top_luxury', 1)
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->whereNotIn('id', $hotelIds)
            ->take(3) // limit results
            ->get();

        return response()->json([
            'status' => 'success',
            'message' => 'Listing',
            'data' => [
                'hotels' => HotelResource::collection($hotels),
                'top_hotels' => HotelResource::collection($topHotels)
            ]
        ], 200);
    }


    public function hotelDetails(Request $r){

        $hotel = Hotel::where('slug', $r->slug)
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->with(['images','rooms.images','faqs','todos','meta'])
            ->withCount(['reviews' => function ($query) {
                $query->where('is_approved', 1);
            }])
            ->withAvg(['reviews' => function ($query) {
                $query->where('is_approved', 1);
            }], 'rating')
            ->firstOrFail();

        // Fetch similar hotels
        $similarHotels = Hotel::where('id', '!=', $hotel->id) // exclude current hotel
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->take(3) // limit results
            ->get();
        return response()->json([
            'status' => 'success',
            'message' => 'Details',
            'data'    => [
                'hotel' => new HotelResource($hotel),
                'similar_hotels' => HotelResource::collection($similarHotels)
            ]
        ]);
    }
}
