<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Http\Resources\PackageResource;
use App\Http\Resources\LocationResource;
use App\Http\Resources\CountryResource;
use App\Http\Resources\PackageLocationResource;
use App\Models\Package;
use App\Models\Location;
use App\Models\Country;
use App\Models\Continent;




class PackageController extends Controller{

    public function exclusivePackages(Request $r){

        $limit     = 4; // default 10 
        $orderBy   = list_config()['order_by']; // default order by id
        $direction = list_config()['direction']; // default desc
        $type      = list_config()['type']; // default National

        $packages = Package::where('is_active', 1)
            ->where('type', $type)
            ->where('is_featured', 1)
            ->where('is_deleted', '!=', 1)
            ->with([
                'images',
                'category',
                'details',
                'location.country.continent',
                'reviews',
            ])
            ->withAvg('reviews', 'rating')
            ->orderBy($orderBy, $direction)
            ->limit($limit)
            ->get();


        return response()->json([
            'status' => 'success',
            'message' => 'Listing',
            'data' => PackageResource::collection($packages)
        ], 200);
    }

    public function topTrendingCountryPackages(Request $r){
        
        $limit     = 4; // default 10 
        $orderBy   = list_config()['order_by']; // default order by id
        $direction = list_config()['direction']; // default desc
        $country   = list_config()['country']; // default India, in feture comming from setting table

        $packages = Package::where('is_active', 1)
            ->where('country_id', 10) // default
            ->where('is_top_trending', 1)
            ->where('is_deleted', '!=', 1)
            ->with(['images', 'category', 'details', 'location.country.continent','reviews'])
            ->withAvg('reviews', 'rating')
            ->orderBy($orderBy, $direction)
            ->limit($limit)
            ->get();

        return response()->json([
            'status' => 'success',
            'message' => 'Listing',
            'data' => PackageResource::collection($packages)
        ], 200);
    }

    public function topTrendingInterNationalPackages(Request $r){

        $limit     = 4; // default 10 
        $orderBy   = list_config()['order_by']; // default order by id
        $direction = list_config()['direction']; // default desc

        // Get all matching packages first
        $packages = Package::where('is_active', 1)
            ->where('type', 2) //  2 = international 
            ->where('is_top_trending', 1)
            ->where('is_deleted', '!=', 1)
            ->with(['images', 'category', 'details', 'location.country.continent','reviews'])
            ->withAvg('reviews', 'rating')
            ->orderBy($orderBy, $direction)
            ->get();

        // Filter so that we keep only unique continent-country-location combos
        $unique = $packages->unique(function ($item) {
            return $item->continent_id . '-' . $item->country_id . '-' . $item->location_id;
        });

        // Take only required limit
        $result = $unique->take($limit)->values();

        return response()->json([
            'status' => 'success',
            'message' => 'Listing',
            'data' => PackageResource::collection($result)
        ], 200);
    }

    public function discoverIndiaTourPackages(Request $r){

        $limit     = 4; // default 10 
        $orderBy   = list_config()['order_by']; // default order by id
        $direction = list_config()['direction']; // default desc

        // Get all India packages (type=1 and country_id =1)
        $packages = Package::where('is_active', 1)
            ->where('country_id', 1) //  India
            ->where('type', 1) //  India
            ->where('is_deleted', '!=', 1)
            ->with(['images', 'category', 'details', 'location.country.continent'])
            ->withAvg('reviews', 'rating')
            ->orderBy($orderBy, $direction)
            ->get();

        // Keep only unique locations (different cities)
        $uniqueByCity = $packages->unique('location_id');

        // Take only required limit (default 4)
        $result = $uniqueByCity->take($limit)->values();

        return response()->json([
            'status' => 'success',
            'message' => 'Listing',
            'data' => PackageResource::collection($result)
        ], 200);
    }

    public function packagesByCountry(Request $r)
    {
        //echo 'msbfhdj';die;
        $limit     = $r->get('limit', list_config()['limit']); // default limit
        $page      = $r->get('page', 1); // current page, default 1
        $orderBy   = list_config()['order_by']; // default order by
        $direction = list_config()['direction']; // default desc

        // Fetch country (single)
        $country = Country::where('slug', $r->slug)
            ->with(['continent', 'details','faqs','meta'])
            ->first();
        if($country) {
            if ($r->exists('package_country')) {
                if(strtolower($r->package_country) == 'international'){
                    if(strtolower($country->name) == 'india'){
                        return invalidRequest();
                    }
                } elseif(strtolower($r->package_country) == 'india') {
                    if(strtolower($country->name) != 'india'){
                        return invalidRequest();
                    }
                } elseif(strtolower($r->package_country) == 'srilanka') {
                    if(strtolower($country->name) != 'sri lanka'){
                        return invalidRequest();
                    }
                }
            }
            //print_r($country);die;
            // Count total packages for this location with filters
            $totalPackages = $country->packages()
                ->where('is_active', 1)
                ->where('is_deleted', '!=', 1)
                ->when($r->filled('category_slug'), function ($query) use ($r) {
                    // filter by category slug from categories table
                    $query->whereHas('category', function ($q) use ($r) {
                        $q->where('slug', $r->category_slug);
                    });
                })
                ->count();

            // Paginated packages with filters
            $packages = $country->packages()
                //->where('type', 2)
                ->where('is_active', 1)
                ->where('is_deleted', '!=', 1)
                ->when($r->filled('category_slug'), function ($query) use ($r) {
                    // filter by category slug from categories table
                    $query->whereHas('category', function ($q) use ($r) {
                        $q->where('slug', $r->category_slug);
                    });
                })
                ->with('details','reviews') // eager load package details
                ->withAvg('reviews', 'rating')
                ->orderBy($orderBy, $direction)
                ->skip(($page - 1) * $limit)
                ->take($limit)
                ->get();

            $categories = \App\Models\Category::whereIn('id', function ($query) use ($country) {
                $query->select('category_id')
                    ->from('packages')
                    ->where('country_id', $country->id)
                    ->where('is_active', 1)
                    ->where('is_deleted', '!=', 1);
            })
            ->select('name', 'slug', 'title') // now choose which fields to return
            ->distinct()
            ->get();

            $sourceLocations = \App\Models\Country::whereIn('id', function ($query) use ($country) {
                $query->select('source_country_id')
                    ->from('packages')
                    ->where('country_id', $country->id) // ✅ current is destination
                    ->where('is_active', 1)
                    ->where('is_deleted', '!=', 1)
                    ->whereNotNull('source_country_id')
                    ->where('source_country_id', '!=', $country->id); // 🚫 exclude self
            })
            ->orWhereIn('id', function ($query) use ($country) {
                $query->select('country_id')
                    ->from('packages')
                    ->where('source_country_id', $country->id) // ✅ current is source
                    ->where('is_active', 1)
                    ->where('is_deleted', '!=', 1)
                    ->whereNotNull('country_id')
                    ->where('country_id', '!=', $country->id); // 🚫 exclude self
            })
            ->where('id', '!=', $country->id) // 🚀 double safety
            ->select('name', 'slug')
            ->distinct()
            ->get();
            return response()->json([
                'status'  => 'success',
                'message' => 'Listing',
                'data'    => [
                    'country'    => new CountryResource($country),
                    'packages'    => PackageResource::collection($packages),
                    'categories'  => $categories,
                    'sourceLocations' => $sourceLocations,  // ✅ unique source locations
                    'pagination'  => [
                        'total' => $totalPackages,
                        'page'  => (int) $page,
                        'limit' => (int) $limit,
                    ]
                ]
            ], 200);
        } else {
            return $this->packagesByLocationNationalInternational($r);
        } 
    }

    public function packagesByLoction(Request $r)
    {
        return $this->packagesByLocationNationalInternational($r);
    }
    

    private function packagesByLocationNationalInternational($r)
    {
        $limit     = $r->get('limit', list_config()['limit']); // default limit
        $page      = $r->get('page', 1); // current page, default 1
        $orderBy   = list_config()['order_by']; // default order by
        $direction = list_config()['direction']; // default desc
        // Fetch location (single)
        $location = Location::where('slug', $r->slug)
            ->where('is_active', 1)
            ->with(['country', 'details', 'faqs','meta'])
            ->first();
        if ($r->exists('package_country')) {
            if(strtolower($r->package_country) == 'international'){
                if(strtolower($location->country->name) == 'india'){
                    return invalidRequest();
                }
            } elseif(strtolower($r->package_country) == 'india') {
                if(strtolower($location->country->name) != 'india'){
                    return invalidRequest();
                }
            } elseif(strtolower($r->package_country) == 'srilanka') {
                if(strtolower($location->country->name) != 'sri lanka'){
                    return invalidRequest();
                }
            }
        }
        // Count total packages for this location with filters
        $totalPackages = $location->packages()
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->when($r->filled('category_slug'), function ($query) use ($r) {
                // filter by category slug from categories table
                $query->whereHas('category', function ($q) use ($r) {
                    $q->where('slug', $r->category_slug);
                });
            })
            ->count();


        // Paginated packages with filters
        $packages = $location->packages()
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->when($r->filled('category_slug'), function ($query) use ($r) {
                // filter by category slug from categories table
                $query->whereHas('category', function ($q) use ($r) {
                    $q->where('slug', $r->category_slug);
                });
            })
            ->with('details','reviews') // eager load package details
            ->withAvg('reviews', 'rating')
            ->orderBy($orderBy, $direction)
            ->skip(($page - 1) * $limit)
            ->take($limit)
            ->get();

        $categories = \App\Models\Category::whereIn('id', function ($query) use ($location) {
            $query->select('category_id')
                ->from('packages')
                ->where('location_id', $location->id)
                ->where('is_active', 1)
                ->where('is_deleted', '!=', 1);
        })
        ->select('name', 'slug', 'title') // now choose which fields to return
        ->distinct()
        ->get();

        $sourceLocations = \App\Models\Location::whereIn('id', function ($query) use ($location) {
            $query->select('source_location_id')
                ->from('packages')
                ->where('location_id', $location->id)
                ->where('is_active', 1)
                ->where('is_deleted', '!=', 1)
                ->whereNotNull('source_location_id'); // avoid nulls
        })
        ->where('id', '!=', $location->id) // 🚀 exclude current location
        ->select('name', 'slug')
        ->distinct()
        ->get();



        return response()->json([
            'status'  => 'success',
            'message' => 'Listing',
            'data'    => [
                'location'    => new LocationResource($location),
                'packages'    => PackageResource::collection($packages),
                'categories'  => $categories,
                'sourceLocations' => $sourceLocations,  // ✅ unique source locations
                'pagination'  => [
                    'total' => $totalPackages,
                    'page'  => (int) $page,
                    'limit' => (int) $limit,
                ]
            ]
        ], 200);
    }

    public function packageDetails(Request $r){

        $package = Package::where('slug', $r->slug)
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->with(['images', 'details','itineraries','location.country.continent','faqs','meta'])
            ->withCount(['reviews' => function ($query) {
                $query->where('is_approved', 1);
            }])
            ->withAvg(['reviews' => function ($query) {
                $query->where('is_approved', 1);
            }], 'rating')
            ->firstOrFail();

        // Fetch similar packages
        $similarPackages = Package::where('id', '!=', $package->id) // exclude current package
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->where('category_id', $package->category_id)
            ->where('country_id', $package->country_id)
            // ->whereHas('details', function ($q) use ($package) {
            //     $q->where('duration_days', $package->details->duration_days)
            //     ->where('duration_nights', $package->details->duration_nights);
            // })
            ->with(['images', 'details', 'location.country.continent','reviews'])
            ->withAvg('reviews', 'rating')
            ->take(3) // limit results
            ->get();
        return response()->json([
            'status' => 'success',
            'message' => 'Details',
            //'data' => new PackageResource($package)
            'data'    => [
                'package' => new PackageResource($package),
                'similar_packages' => PackageResource::collection($similarPackages)
            ]
        ]);
    }

    public function internationalCountries(Request $r)
    {
        $limit     = (int) $r->get('limit', 6);
        $orderBy   = $r->get('order_by', list_config()['order_by']);
        $direction = $r->get('direction', list_config()['direction']);
        $continent = $r->get('continent'); // optional filter by slug
        $page      = (int) $r->get('page', 1);

        $query = Package::where('is_active', 1)
            ->where('type', 2) // International
            ->where('is_deleted', '!=', 1)
            ->with(['country.details', 'country.continent']);

        // Filter only if continent param is passed
        if ($r->has('continent')) {
            // If param is empty or invalid slug → return empty
            if (! $r->filled('continent') || ! Continent::where('slug', $continent)->exists()) {
                return response()->json([
                    'status'  => 'success',
                    'message' => 'Listing',
                    'data'    => [
                        'countries'  => [],
                        'pagination' => [
                            'total' => 0,
                            'limit' => $limit,
                            'page'  => $page,
                            'last_page' => 0,
                        ],
                    ],
                ], 200);
            }

            // Valid slug → filter by continent
            $query->whereHas('country.continent', function ($q) use ($continent) {
                $q->where('slug', $continent);
            });
        }

        // Fetch all matched packages
        $packages = $query->orderBy($orderBy, $direction)->get();

        // Extract unique countries
        $countries = $packages->pluck('country')->unique('id')->values();

        // Pagination
        $total = $countries->count();
        $paged = $countries->forPage($page, $limit)->values();

        return response()->json([
            'status'  => 'success',
            'message' => 'Listing',
            'data'    => [
                'countries'  => CountryResource::collection($paged),
                'pagination' => [
                    'total'      => $total,
                    'limit'      => $limit,
                    'page'       => $page
                ],
            ],
        ], 200);
    }


    /***************************Function will remove****************************** */
    public function internationalPackagesByCountry(Request $r)
    {
        //echo 'msbfhdj';die;
        $limit     = $r->get('limit', list_config()['limit']); // default limit
        $page      = $r->get('page', 1); // current page, default 1
        $orderBy   = list_config()['order_by']; // default order by
        $direction = list_config()['direction']; // default desc

        // Fetch country (single)
        $country = Country::where('slug', $r->get('country'))
            ->with(['continent', 'details','faqs'])
            ->firstOrFail();
        //print_r($country);die;
        // Count total packages for this location with filters
        $totalPackages = $country->packages()
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->count();

        // Paginated packages with filters
        $packages = $country->packages()
            ->where('type', 2)
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->with('details','reviews') // eager load package details
            ->withAvg('reviews', 'rating')
            ->orderBy($orderBy, $direction)
            ->skip(($page - 1) * $limit)
            ->take($limit)
            ->get();

        return response()->json([
            'status'  => 'success',
            'message' => 'Listing',
            'data'    => [
                'country'    => new CountryResource($country),
                'packages'    => PackageResource::collection($packages),
                'pagination'  => [
                    'total' => $totalPackages,
                    'page'  => (int) $page,
                    'limit' => (int) $limit,
                ]
            ]
        ], 200);
    }
}
