<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Yajra\DataTables\Facades\DataTables;

class EmployeeController extends Controller
{
    public function addView()
    {
        $users = User::whereIn('role_id', [2, 4])->get();
        return view('auth.employee.add', compact('users'));
    }
    public function addEmployee(Request $request)
    {
        // return $request;
        $validator = Validator::make($request->all(), [
            'name'            => 'required|string|max:255',
            'employee_code'   => 'string|max:255|unique:users,employee_code',
            'role_id'         => 'required|in:1,2,3,4',
            'reporting_to'          => 'nullable|array|min:1',
            'reporting_to.*'        => 'nullable|string',
            'monthly_rate'     => 'nullable|numeric|min:0',
            'hourly_rateINR'     => 'nullable|numeric|min:0',
            'hourly_rateUSD'     => 'nullable|numeric|min:0',
            'experience'      => 'nullable',
            'password'        => 'nullable|string|min:6',
        ], [
            'name.required' => 'Name is required.',
            'role_id.required' => 'Please select a role.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        $user = User::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'role_id'  => $request->role_id,
            'password' => Hash::make($request->password ?? '123'),
            'employee_code'  => $request->employee_code,
            'department'  => $request->department_id ?? null,
            'phone'    => $request->phone,
            // 'status'   => 'active',
            'reporting_to' => $request->has('reporting_to') ? json_encode($request->reporting_to) : null,
        ]);


        // Employee::create([
        //     'user_id'        => $user->id,
        //     'skill_set'         => json_encode($request->skills) ?? null,
        //     'hourly_rate'    => $request->hourly_rate ?? null,
        //     'experience'     => $request->experience ?? null,
        //     // 'status'   => 'available',
        //     'monthly_rate'     => $request->monthly_rate ?? null,
        //     'hourly_rateINR'     => $request->hourly_rateINR ?? null,
        //     'hourly_rateUSD'     => $request->hourly_rateUSD ?? null,
        // ]);

        return redirect()->route('employee.manage.view')->with('success', 'Employee added successfully.');
    }
    public function manageView()
    {
        return view('auth.employee.manage');
    }
    public function getEmployees(Request $request)
    {
        $data = User::all();
        return DataTables::of($data)
            ->addColumn('id', function ($row) {
                return $row->id;
            })
            // ->editColumn('skill_set', function ($row) {
            //     $skills = json_decode($row->skill_set, true); // decode to array
            //     $badges = '';
            //     if (is_array($skills)) {
            //         foreach ($skills as $skill) {
            //             $badges .= '<span class="badge bg-primary me-1">' . e($skill) . '</span>';
            //         }
            //     }
            //     return $badges;
            // })
            ->addColumn('action', function ($row) {
                $editUrl = route('employees.edit', $row->id);
                $deleteUrl = route('employees.destroy', $row->id);

                return '
            <a href="' . $editUrl . '" class="btn btn-sm btn-warning me-1">Edit</a>
            <button data-id="' . $row->id . '" class="btn btn-sm btn-danger delete-btn">Delete</button>
        ';
            })
            ->rawColumns(['action'])
            ->make(true);
    }
    public function edit($id)
    {
        $employee = User::findOrFail($id);
        $users = User::whereIn('role_id', [2, 4])->get();
        return view('auth.employee.edit', compact('employee', 'users'));
    }
    public function editSubmit(Request $request, $id)
    {
        // return $request;
        $user = User::findOrFail($id);
        // $user = $employee->user;

        $validator = Validator::make($request->all(), [
            'name'            => 'required|string|max:255',
            'employee_code'   => [
                'nullable',
                'string',
                'max:255',
                Rule::unique('users', 'employee_code')->ignore($user->id),
            ],
            'role_id'         => 'required',
            // 'email'           => [
            //     'email',
            //     Rule::unique('users', 'email')->ignore($user->id),
            // ],
        ], [
            'name.required' => 'Name is required.',
            'role_id.required' => 'Please select a role.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user->update([
            'name'           => $request->filled('name') ? $request->name : $user->name,
            'email'          => $request->filled('email') ? $request->email : $user->email,
            'employee_code'  => $request->filled('employee_code') ? $request->employee_code : $user->employee_code,
            'role_id'        => $request->filled('role_id') ? $request->role_id : $user->role_id,
            'department'     => $request->filled('department_id') ? $request->department_id : $user->department,
            'phone'          => $request->filled('phone') ? $request->phone : $user->phone,
            'password'       => $request->filled('password') ? Hash::make($request->password) : $user->password,
            'reporting_to'   => $request->filled('reporting_to') ? json_encode($request->reporting_to) : $user->reporting_to,
        ]);

        // $employee->update([
        //     'skill_set'      => $request->filled('skills') ? json_encode($request->skills) : $employee->skill_set,
        //     // 'hourly_rate'    => $request->filled('hourly_rate') ? $request->hourly_rate : $employee->hourly_rate,
        //     'experience'     => $request->filled('experience') ? $request->experience : $employee->experience,
        //     'monthly_rate'   => $request->filled('monthly_rate') ? $request->monthly_rate : $employee->monthly_rate,
        //     'hourly_rateINR' => $request->filled('hourly_rateINR') ? $request->hourly_rateINR : $employee->hourly_rateINR,
        //     'hourly_rateUSD' => $request->filled('hourly_rateUSD') ? $request->hourly_rateUSD : $employee->hourly_rateUSD,
        // ]);


        return redirect()->route('employee.manage.view')->with('success', 'Employee updated successfully.');
    }

    public function destroy($id)
    {
        $employee = Employee::findOrFail($id);
        if ($employee->user_id) {
            $employee->user()->delete();
        }
        $employee->delete();
        return redirect()->route('employee.manage.view')->with('error', 'Employee deleted successfully.');
    }
}
