<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

use App\Models\SchoolClass;
use App\Models\SchoolSection;

class ClassSetupController extends Controller
{
    public function index()
    {
        $classes = SchoolClass::with(['sections', 'students'])->get();

        return view('class.index', compact('classes'));
    }

    public function addClass(Request $request)
    {
        $request->validate([
            'class_name' => 'required|unique:classes,class_name'
        ]);

        SchoolClass::create([
            'class_name' => $request->class_name
        ]);

        return back()->with('success', 'Class added successfully.');
    }


    public function updateClass(Request $request, $id)
    {
        $class = SchoolClass::findOrFail($id);

        $request->validate([
            'class_name' => 'required|string|max:255|unique:classes,class_name,' . $id,
        ]);

        $class->update([
            'class_name' => $request->class_name,
        ]);

        return response()->json(['success' => true]);
    }


    public function deleteClass($id)
    {
        SchoolClass::findOrFail($id)->delete();

        return response()->json(['success' => true]);
    }

    

    public function addSection(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'class_id'     => 'required',
            'section_name' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        // 🔒 Class-wise duplicate check
        $exists = SchoolSection::where('class_id', $request->class_id)
            ->where('section_name', $request->section_name)
            ->exists();

        if ($exists) {
            return response()->json([
                'errors' => [
                    'section_name' => ['This section already exists for the selected class']
                ]
            ], 422);
        }

        $section = SchoolSection::create([
            'class_id'     => $request->class_id,
            'section_name' => $request->section_name,
            'status'       => 1
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Section added successfully',
            'data'    => $section
        ]);
    }



    public function deleteSection($id)
    {
        SchoolSection::findOrFail($id)->delete();

        return response()->json(['success' => true]);
    }
}
