<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Staff;
use App\Models\StaffSalary;
use App\Models\AcademicYear;
use App\Models\StaffBankDetails;
use App\Models\StaffAcademicYear;
use App\Models\StaffMonthlySalary;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class StaffController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /* =========================================================
     * INDEX – show only staff of ACTIVE academic year
     * ========================================================= */
    public function index(Request $request)
    {
        $tab = $request->get('tab','list');
        session(['tab' => $tab]);

        $academic_years = AcademicYear::where('status',1)->get();
        $activeAcademicYearId = $request->financial_year_id
            ?? AcademicYear::where('status',1)->value('id');

        /* ---------------- STAFF LIST TAB ---------------- */
        $staffs = Staff::when($tab === 'staff', function ($q) use ($request) {
                $q->when($request->filled('name'), fn($qq) =>
                    $qq->where('name','like','%'.$request->name.'%')
                )
                ->when($request->filled('category'), fn($qq) =>
                    $qq->where('category',$request->category)
                );
            })
            ->latest()
            ->paginate(20)
            ->withQueryString();

        /* ---------------- SALARY TAB ---------------- */
        $salaryStaffs = collect();

        if ($tab === 'staff_salary' && $request->filled('month')) {

            $month = $request->month; // YYYY-MM

            $salaryStaffs = Staff::with([
                    'salary' => function ($q) use ($activeAcademicYearId) {
                        $q->where('financial_year_id',$activeAcademicYearId)
                        ->where('status',1);
                    },
                    'monthlySalaries' => function ($q) use ($month, $activeAcademicYearId) {
                        $q->where('pay_for_month',$month)
                        ->where('financial_year_id',$activeAcademicYearId);
                    }
                ])
                ->whereHas('salary', function ($q) use ($activeAcademicYearId) {
                    $q->where('financial_year_id',$activeAcademicYearId)
                    ->where('status',1);
                })
                ->when($request->filled('category'), fn($q) =>
                    $q->where('category',$request->category)
                )
                ->where('status',1)
                ->orderBy('name')
                ->get()
                ->map(function ($staff) use ($month) {

                    $monthly = $staff->monthlySalaries->first();
                    $baseSalary = $staff->salary->first();

                    if ($monthly) {
                        // 🔹 PAID
                        return [
                            'staff'      => $staff,
                            'salary'     => $monthly,
                            'status'     => 'paid',
                            'source'     => 'monthly',
                        ];
                    }

                    // 🔹 PENDING
                    return [
                        'staff'      => $staff,
                        'salary'     => $baseSalary,
                        'status'     => 'pending',
                        'source'     => 'base',
                    ];
                });
        }

        return view('staff.index', compact(
            'staffs',
            'salaryStaffs',
            'academic_years',
            'activeAcademicYearId'
        ));
    }



    /* =========================================================
     * STORE
     * ========================================================= */
    public function store(Request $request)
    {
        session(['tab' => 'add_staff']);

        $validated = $request->validate([
            // STAFF
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'required|digits_between:10,15',
            'address' => 'nullable|string',
            'category' => 'required|string',
            'designation' => 'required|string',
            'qualification' => 'required|string',
            'join_date' => 'required|date',
            'photo' => 'nullable|image|max:2048',

            // SALARY
            'basic' => 'required|numeric|min:0',
            'hra' => 'nullable|numeric|min:0',
            'da' => 'nullable|numeric|min:0',
            'ta' => 'nullable|numeric|min:0',
            'medical_allowance' => 'nullable|numeric|min:0',
            'conveyance' => 'nullable|numeric|min:0',
            'washing_allowance' => 'nullable|numeric|min:0',
            'special_allowance' => 'nullable|numeric|min:0',

            'pf' => 'nullable|numeric|min:0',
            'esi' => 'nullable|numeric|min:0',
            'p_tax' => 'nullable|numeric|min:0',
            'tds' => 'nullable|numeric|min:0',
            'others_deduction' => 'nullable|numeric|min:0',

            'employer_pf' => 'nullable|numeric|min:0',
            'employer_esi' => 'nullable|numeric|min:0',

            'financial_year_id' => 'required|integer',

            // BANK
            'bank_name' => 'nullable|string|max:255',
            'branch_name' => 'nullable|string|max:255',
            'account_no' => 'nullable|string|max:30',
            'account_holder' => 'nullable|string|max:255',
            'ifsc' => 'nullable|string|max:20',
        ]);

        DB::transaction(function () use ($request, $validated) {
            $profileImage = $request->file('photo')
                ? $request->file('photo')->store('staff/profile', 'public')
                : null;

            $activeAcademicYearId = AcademicYear::where('status', 1)->value('id');

            /* ---------- STAFF ---------- */
            $staff = Staff::create([
                'name' => $validated['name'],
                'email' => $validated['email'] ?? null,
                'phone' => $validated['phone'],
                'address' => $validated['address'] ?? null,
                'category' => $validated['category'],
                'designation' => $validated['designation'],
                'qualification' => $validated['qualification'],
                'join_date' => $validated['join_date'],
                'photo' => $profileImage,
                'status' => 1,
            ]);

            /* ---------- STAFF ACADEMIC YEAR ---------- */
            StaffAcademicYear::create([
                'staff_id' => $staff->id,
                'financial_year_id' => $activeAcademicYearId,
                'status' => 1,
            ]);

            /* ---------- SALARY CALC ---------- */
            $totalEarnings =
                $request->basic +
                ($request->hra ?? 0) +
                ($request->da ?? 0) +
                ($request->ta ?? 0) +
                ($request->medical_allowance ?? 0) +
                ($request->conveyance ?? 0) +
                ($request->washing_allowance ?? 0) +
                ($request->special_allowance ?? 0);

            $totalDeductions =
                ($request->pf ?? 0) +
                ($request->esi ?? 0) +
                ($request->p_tax ?? 0) +
                ($request->tds ?? 0) +
                ($request->others_deduction ?? 0);

            $netSalary = $totalEarnings - $totalDeductions;

            $ctc = $netSalary +
                ($request->employer_pf ?? 0) +
                ($request->employer_esi ?? 0);

            /* ---------- STAFF SALARY ---------- */
            StaffSalary::create([
                'staff_id' => $staff->id,
                'basic' => $request->basic,
                'hra' => $request->hra ?? 0,
                'da' => $request->da ?? 0,
                'ta' => $request->ta ?? 0,
                'medical_allowance' => $request->medical_allowance ?? 0,
                'conveyance' => $request->conveyance ?? 0,
                'washing_allowance' => $request->washing_allowance ?? 0,
                'special_allowance' => $request->special_allowance ?? 0,
                'total_earnings' => $totalEarnings,
                'pf' => $request->pf ?? 0,
                'esi' => $request->esi ?? 0,
                'p_tax' => $request->p_tax ?? 0,
                'tds' => $request->tds ?? 0,
                'others_deduction' => $request->others_deduction ?? 0,
                'total_deductions' => $totalDeductions,
                'net_salary' => $netSalary,
                'employer_pf' => $request->employer_pf ?? 0,
                'employer_esi' => $request->employer_esi ?? 0,
                'ctc' => $ctc,
                'financial_year_id' => $request->financial_year_id,
                'status' => 1,
            ]);

            /* ---------- BANK ---------- */
            if ($request->bank_name || $request->account_no) {
                StaffBankDetails::create([
                    'staff_id' => $staff->id,
                    'bank_name' => $request->bank_name,
                    'branch_name' => $request->branch_name,
                    'acc_no' => $request->account_no,
                    'acc_holder_name' => $request->account_holder,
                    'ifsc_code' => $request->ifsc,
                ]);
            }
        });

        return redirect()
            ->route('staff.index')
            ->with('tab', 'list')
            ->with('success', 'Staff added successfully');
    }

    /* =========================================================
     * EDIT
     * ========================================================= */
    public function edit($id)
    {
        $staff = Staff::with(['salary', 'bankDetails'])->findOrFail($id);
        $academic_years = AcademicYear::where('status', 1)->get();
        return view('staff.edit', compact('staff','academic_years'));
    }

    /* =========================================================
     * UPDATE
     * ========================================================= */
    public function update(Request $request, $id)
    {
        $staff = Staff::findOrFail($id);
        /**
         * ==================================================
         * CASE 1: STATUS ONLY UPDATE (TOGGLE)
         * ==================================================
         */
        if ($request->has('status')) {

            $staff->update([
                'status' => $request->boolean('status')
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Staff status updated successfully'
            ]);
        }

        /**
         * ==================================================
         * CASE 2: FULL UPDATE
         * ==================================================
         */

        $validated = $request->validate([
            // STAFF
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'required|digits_between:10,15',
            'address' => 'nullable|string',
            'category' => 'required|string',
            'designation' => 'required|string',
            'qualification' => 'required|string',
            'join_date' => 'required|date',
            'photo' => 'nullable|image|max:2048',

            // SALARY
            'basic' => 'required|numeric|min:0',
            'hra' => 'nullable|numeric|min:0',
            'da' => 'nullable|numeric|min:0',
            'ta' => 'nullable|numeric|min:0',
            'medical_allowance' => 'nullable|numeric|min:0',
            'conveyance' => 'nullable|numeric|min:0',
            'washing_allowance' => 'nullable|numeric|min:0',
            'special_allowance' => 'nullable|numeric|min:0',

            'pf' => 'nullable|numeric|min:0',
            'esi' => 'nullable|numeric|min:0',
            'p_tax' => 'nullable|numeric|min:0',
            'tds' => 'nullable|numeric|min:0',
            'others_deduction' => 'nullable|numeric|min:0',

            'employer_pf' => 'nullable|numeric|min:0',
            'employer_esi' => 'nullable|numeric|min:0',

            // BANK
            'bank_name' => 'nullable|string|max:255',
            'branch_name' => 'nullable|string|max:255',
            'account_no' => 'nullable|string|max:30',
            'account_holder' => 'nullable|string|max:255',
            'ifsc' => 'nullable|string|max:20',
        ]);

        DB::transaction(function () use ($request, $validated, $staff) {

            if ($request->hasFile('photo')) {
                $staff->photo =
                    $request->file('photo')->store('staff/profile', 'public');
            }

            /* ---------- STAFF UPDATE ---------- */
            $staff->update([
                'name' => $validated['name'],
                'email' => $validated['email'] ?? null,
                'phone' => $validated['phone'],
                'address' => $validated['address'] ?? null,
                'category' => $validated['category'],
                'designation' => $validated['designation'],
                'qualification' => $validated['qualification'],
                'join_date' => $validated['join_date'],
            ]);

            /* ---------- SALARY RECALC ---------- */
            $totalEarnings =
                $request->basic +
                ($request->hra ?? 0) +
                ($request->da ?? 0) +
                ($request->ta ?? 0) +
                ($request->medical_allowance ?? 0) +
                ($request->conveyance ?? 0) +
                ($request->washing_allowance ?? 0) +
                ($request->special_allowance ?? 0);

            $totalDeductions =
                ($request->pf ?? 0) +
                ($request->esi ?? 0) +
                ($request->p_tax ?? 0) +
                ($request->tds ?? 0) +
                ($request->others_deduction ?? 0);

            $netSalary = $totalEarnings - $totalDeductions;

            $ctc =
                $netSalary +
                ($request->employer_pf ?? 0) +
                ($request->employer_esi ?? 0);

            /* ---------- SALARY UPDATE ---------- */
            $staff->salary()->update([
                'basic' => $request->basic,
                'hra' => $request->hra ?? 0,
                'da' => $request->da ?? 0,
                'ta' => $request->ta ?? 0,
                'medical_allowance' => $request->medical_allowance ?? 0,
                'conveyance' => $request->conveyance ?? 0,
                'washing_allowance' => $request->washing_allowance ?? 0,
                'special_allowance' => $request->special_allowance ?? 0,
                'total_earnings' => $totalEarnings,

                'pf' => $request->pf ?? 0,
                'esi' => $request->esi ?? 0,
                'p_tax' => $request->p_tax ?? 0,
                'tds' => $request->tds ?? 0,
                'others_deduction' => $request->others_deduction ?? 0,
                'total_deductions' => $totalDeductions,

                'net_salary' => $netSalary,
                'employer_pf' => $request->employer_pf ?? 0,
                'employer_esi' => $request->employer_esi ?? 0,
                'ctc' => $ctc,
            ]);

            /* ---------- BANK UPDATE ---------- */
            if ($staff->bankDetails) {
                $staff->bankDetails->update([
                    'bank_name' => $request->bank_name,
                    'branch_name' => $request->branch_name,
                    'acc_no' => $request->account_no,
                    'acc_holder_name' => $request->account_holder,
                    'ifsc_code' => $request->ifsc,
                ]);
            }
        });

        return redirect()
            ->route('staff.index')
            ->with('success', 'Staff updated successfully');
    }

    public function show($id)
    {
        $staff = Staff::with([
            'salary.financialYear',
            'bankDetails',
            'academicYears.financialYear'
        ])->findOrFail($id);

        return view('staff.show', compact('staff'));
    }



    /* =========================================================
     * DELETE (Soft logic via academic year)
     * ========================================================= */
    public function destroy($id)
    {
        $activeAcademicYearId = AcademicYear::where('status', 1)->value('id');

        StaffAcademicYear::where('staff_id', $id)
            ->where('financial_year_id', $activeAcademicYearId)
            ->update(['status' => 0]);

        return redirect()->route('staff.index')->with('success', 'Staff removed from current academic year');
    }




    public function paySalary(Request $request)
    {
        $request->validate([
            'staff_id' => 'required|integer',
            'financial_year_id' => 'required|integer',
            'month' => 'required|date_format:Y-m', // 👈 IMPORTANT
        ]);

        DB::transaction(function () use ($request) {

            //  Prevent duplicate payment
            $alreadyPaid = StaffMonthlySalary::where('staff_id', $request->staff_id)
                ->where('pay_for_month', $request->month)
                ->where('financial_year_id', $request->financial_year_id)
                ->exists();

            if ($alreadyPaid) {
                throw new \Exception('Salary already paid for this month');
            }

            $salary = StaffSalary::where('staff_id', $request->staff_id)
                ->where('financial_year_id', $request->financial_year_id)
                ->where('status', 1)
                ->firstOrFail();

            StaffMonthlySalary::create([
                'staff_id' => $salary->staff_id,

                // earnings
                'basic' => $salary->basic,
                'hra' => $salary->hra,
                'medical_allowance' => $salary->medical_allowance,
                'conveyance' => $salary->conveyance,
                'washing_allowance' => $salary->washing_allowance,
                'special_allowance' => $salary->special_allowance,
                'total_earnings' => $salary->total_earnings,

                // deductions
                'pf' => $salary->pf,
                'esi' => $salary->esi,
                'p_tax' => $salary->p_tax,
                'tds' => $salary->tds,
                'advance' => 0,
                'others_deduction' => $salary->others_deduction,
                'total_deductions' => $salary->total_deductions,

                // net
                'net_salary' => $salary->net_salary,
                'employer_pf' => $salary->employer_pf,
                'employer_esi' => $salary->employer_esi,
                'ctc' => $salary->ctc,

                // ✅ FIXED HERE
                'pay_for_month' => $request->month, // e.g. 2025-11
                'paid_date' => now(),
                'financial_year_id' => $request->financial_year_id,
            ]);
        });

        return back()
            ->with('tab', 'staff_salary')
            ->with('success', 'Salary paid successfully');
    }


    public function salarySlip($staffId, $month)
    {
        $slip = StaffMonthlySalary::with(['staff', 'financialYear'])
            ->where('staff_id', $staffId)
            ->where('pay_for_month', $month)
            ->firstOrFail();

        return view('staff.salary-slip', compact('slip'));
    }

    public function payAllPendingSalaries(Request $request)
    {
        $month = $request->month;
        $financialYearId = $request->financial_year_id;

        $pendingStaffs = Staff::with(['salary' => function($q) use ($financialYearId) {
                $q->where('financial_year_id', $financialYearId)
                ->where('status', 1);
            },
            'monthlySalaries' => function($q) use ($month, $financialYearId) {
                $q->where('pay_for_month', $month)
                ->where('financial_year_id', $financialYearId);
            }
        ])
        ->whereHas('salary', function($q) use ($financialYearId) {
            $q->where('financial_year_id', $financialYearId)
            ->where('status', 1);
        })
        ->get()
        ->filter(function($staff) {
            return $staff->monthlySalaries->isEmpty(); // only pending
        });

        DB::transaction(function() use ($pendingStaffs, $month, $financialYearId) {
            foreach($pendingStaffs as $staff) {
                $salary = $staff->salary->first();
                if ($salary) {
                    \App\Models\StaffMonthlySalary::create([
                        'staff_id' => $staff->id,

                        // Earnings
                        'basic' => $salary->basic,
                        'hra' => $salary->hra,
                        'medical_allowance' => $salary->medical_allowance,
                        'conveyance' => $salary->conveyance,
                        'washing_allowance' => $salary->washing_allowance,
                        'special_allowance' => $salary->special_allowance,
                        'total_earnings' => $salary->total_earnings,

                        // Deductions
                        'pf' => $salary->pf,
                        'esi' => $salary->esi,
                        'p_tax' => $salary->p_tax,
                        'tds' => $salary->tds,
                        'advance' => 0,
                        'others_deduction' => $salary->others_deduction,
                        'total_deductions' => $salary->total_deductions,

                        // Net & Employer
                        'net_salary' => $salary->net_salary,
                        'employer_pf' => $salary->employer_pf,
                        'employer_esi' => $salary->employer_esi,
                        'ctc' => $salary->ctc,

                        'pay_for_month' => $month,
                        'paid_date' => now(),
                        'financial_year_id' => $financialYearId,
                    ]);
                }
            }
        });

        return back()->with('success', 'All pending salaries have been paid successfully')
        ->with('tab', 'staff_salary');
    }



}
