<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

use App\Models\Student;
use App\Models\ParentModel;
use App\Models\StudentClassDetail;
use App\Models\SchoolClass;
use App\Models\SchoolSection;
use App\Models\AcademicYear;
use App\Models\SchoolSession;


class StudentController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /*public function index(Request $request)
    {
        //$students = Student::with('classDetails')->latest()->paginate(25);
        
        $students = Student::with([
            'activeClass.class' => fn ($q) => $q->where('status', 1),
            'activeClass.section',
        ])
        ->when(request('q'), function ($query) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . request('q') . '%')
                ->orWhere('admission_no', 'like', '%' . request('q') . '%');
            });
        })
        ->when(request('class_id'), function ($query) {
            $query->whereHas('activeClass', function ($q) {
                $q->where('class_id', request('class_id'));
            });
        })
        ->when(request('section_id'), function ($query) {
            $query->whereHas('activeClass', function ($q) {
                $q->where('section_id', request('section_id'));
            });
        })
        ->latest('id')
        ->paginate(25)
        ->withQueryString();


        $classes = SchoolClass::with('sections')->where('status',1)->get();
        $academic_years = AcademicYear::where('status',1)->get();
        return view('students.index', compact('students','classes','academic_years'));
    }*/

    public function index(Request $request)
    {
        $classes = SchoolClass::where('status', 1)->get();
        $sessions = SchoolSession::where('status', 1)->get();

        $sections = [];
        if ($request->class_id) {
            $sections = SchoolSection::where('class_id', $request->class_id)->get();
        }

        $activeAcademicYearId = AcademicYear::where('status', 1)->value('id');
        $academic_years = AcademicYear::where('status', 1)->get();


        $students = Student::with(['activeClass.class', 'activeClass.section'])
            ->whereHas('activeClass', function ($q) use ($activeAcademicYearId) {
                $q->where('status', 1)
                ->where('financial_year_id', $activeAcademicYearId);
            })
            ->when($request->q, function ($q) use ($request) {
                $q->where(function ($x) use ($request) {
                    $x->where('name', 'like', "%{$request->q}%")
                    ->orWhere('admission_no', 'like', "%{$request->q}%");
                });
            })
            ->when($request->class_id, function ($q) use ($request) {
                $q->whereHas('activeClass', function ($x) use ($request, $activeAcademicYearId) {
                    $x->where('class_id', $request->class_id)
                    ->where('financial_year_id', $activeAcademicYearId);
                });
            })
            ->when($request->section_id, function ($q) use ($request) {
                $q->whereHas('activeClass', function ($x) use ($request) {
                    $x->where('section_id', $request->section_id);
                });
            })
            ->latest()
            ->paginate(25)
            ->withQueryString();

        return view('students.index', compact(
            'students',
            'classes',
            'sections',
            'academic_years',
            'sessions'
        ));
    }

    public function search(Request $request)
    {
        $q = $request->q;

        return Student::query()
            ->where('status', 1)
            ->where(function ($query) use ($q) {
                $query->where('name', 'like', "%$q%")
                    ->orWhere('admission_no', 'like', "%$q%");
            })
            ->with(['activeClass.class', 'activeClass.section'])
            ->limit(20)
            ->get()
            ->map(function ($s) {
                return [
                    'id'            => $s->id,
                    'admission_no'  => $s->admission_no,
                    'name'          => $s->name,
                    'class_name'    => optional($s->activeClass->class)->class_name,
                    'section_name'  => optional($s->activeClass->section)->section_name,
                    'roll'  => $s->activeClass->roll_no,
                ];
            });
    }



    public function store(Request $request)
    {
        $validated = $request->validate([
            // ================= STUDENT =================
            'name' => 'required|string|max:255',
            //'admission_no' => 'required|string|max:100|unique:students,admission_no',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female,Other',
            'nationality' => 'nullable|string|max:100',
            'mother_tongue' => 'nullable|string|max:100',
            'category' => 'required|in:GEN,OBC,SC,ST',
            'cast' => 'nullable|string|max:100',
            'address' => 'required|string',
            'local_address' => 'required|string',

            'belongs_to_minority_group' => 'nullable|boolean',
            'is_disability_ph' => 'nullable|boolean',
            'is_transport_required' => 'nullable|boolean',

            'profile_image' => 'nullable|image|max:2048',
            'dob_photo_copy' => 'nullable|mimes:jpg,jpeg,png,pdf|max:2048',

            // ================= PARENT / GUARDIAN =================
            'father_name' => 'required|string|max:255',
            'mother_name' => 'required|string|max:255',

            'father_qualification' => 'nullable|string|max:255',
            'mother_qualification' => 'nullable|string|max:255',

            'father_occupation' => 'required|string|max:255',
            'mother_occupation' => 'required|string|max:255',

            'father_phone' => 'required|digits:10',
            'mother_phone' => 'required|digits:10',

            'father_voter_id' => 'nullable|string|max:50',
            'mother_voter_id' => 'nullable|string|max:50',

            'guardian_name' => 'nullable|string|max:255',
            'guardian_phone' => 'nullable|digits:10',
            'guardian_address' => 'nullable|string',

            // ================= CLASS =================
            'class_id' => 'required|integer',
            'section_id' => 'required|integer',
            //'roll_no' => 'required|integer',
            'session_id' => 'required|integer',
            'financial_year' => 'required|integer',
        ]);

        DB::transaction(function () use ($request, $validated) {

            // ================= FILE UPLOAD =================
            $profileImage = $request->file('profile_image')
                ? $request->file('profile_image')->store('students/profile', 'public')
                : null;

            $dobCopy = $request->file('dob_photo_copy')
                ? $request->file('dob_photo_copy')->store('students/dob', 'public')
                : null;

            // ================= PARENT =================
            $parent = ParentModel::create([
                'father_name' => $validated['father_name'],
                'mother_name' => $validated['mother_name'],

                'father_qualification' => $validated['father_qualification'] ?? null,
                'mother_qualification' => $validated['mother_qualification'] ?? null,

                'father_occupation' => $validated['father_occupation'],
                'mother_occupation' => $validated['mother_occupation'],

                'father_phone' => $validated['father_phone'],
                'mother_phone' => $validated['mother_phone'],

                'father_voter_id' => $validated['father_voter_id'] ?? null,
                'mother_voter_id' => $validated['mother_voter_id'] ?? null,

                'guardian_name' => $validated['guardian_name'] ?? null,
                'guardian_phone' => $validated['guardian_phone'] ?? null,
                'guardian_address' => $validated['guardian_address'] ?? null,
            ]);

            // ================= ADMISSION NO (AUTO) =================
            $lastStudentId = Student::max('id') ?? 0;
            $admission_no = 'ADM-' . date('Y') . '-' . str_pad($lastStudentId + 1, 5, '0', STR_PAD_LEFT);

            // ================= STUDENT =================
            $student = Student::create([
                'name' => $validated['name'],
                'admission_no' => $admission_no,
                'dob' => $validated['dob'],
                'gender' => $validated['gender'],
                'nationality' => $validated['nationality'],
                'mother_tongue' => $validated['mother_tongue'],
                'category' => $validated['category'],
                'cast' => $validated['cast'],
                'address' => $validated['address'],
                'local_address' => $validated['local_address'],

                'belongs_to_minority_group' => $request->boolean('belongs_to_minority_group'),
                'is_disability_ph' => $request->boolean('is_disability_ph'),
                'is_transport_required' => $request->boolean('is_transport_required'),

                'profile_image' => $profileImage,
                'dob_photo_copy' => $dobCopy,

                'parent_id' => $parent->id,
                'status' => 1,
            ]);

            
            // ================= NEXT ROLL NO (CLASS + SECTION + YEAR) =================
            $maxRoll = StudentClassDetail::where('class_id', $validated['class_id'])
                ->where('section_id', $validated['section_id'])
                ->where('financial_year_id', $validated['financial_year_id'])
                ->where('status', 1)
                ->max('roll_no');

            $nextRollNo = $maxRoll ? $maxRoll + 1 : 1;

            // ================= CLASS DETAILS =================
            StudentClassDetail::create([
                'student_id' => $student->id,
                'class_id' => $validated['class_id'],
                'section_id' => $validated['section_id'],
                'roll_no' => $nextRollNo,
                'session_id' => $validated['session_id'],
                'financial_year_id' => $validated['financial_year_id'],
                'status' => 1,
                'start_date' => now(),
            ]);
            
        });


        return redirect()
            ->route('students.index')
            ->with('success', 'Student registered successfully');
    }

    public function edit($id)
    {
        $student = Student::with([
            'parent',
            'classDetails' => function ($q) {
                $q->where('status', 1);
            }
        ])->findOrFail($id);

        $classes = SchoolClass::where('status', 1)->with('sections')->get();
        $academicYears = AcademicYear::where('status', 1)->get();
        $sessions = SchoolSession::where('status', 1)->get();

        return view('students.edit', compact(
            'student',
            'classes',
            'academicYears',
            'sessions'
        ));
    }

    public function update(Request $request, $id)
    {
        $student = Student::with('parent', 'classDetails')->findOrFail($id);

        /**
         * ==================================================
         * CASE 1: STATUS ONLY UPDATE (TOGGLE)
         * ==================================================
         */
        if ($request->has('status')) {

            $student->update([
                'status' => $request->boolean('status')
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Student status updated successfully'
            ]);
        }

        /**
         * ==================================================
         * CASE 2: FULL UPDATE
         * ==================================================
         */
        $validated = $request->validate([
            // ================= STUDENT =================
            'name' => 'required|string|max:255',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female,Other',
            'nationality' => 'nullable|string|max:100',
            'mother_tongue' => 'nullable|string|max:100',
            'category' => 'required|in:GEN,OBC,SC,ST',
            'cast' => 'nullable|string|max:100',
            'address' => 'required|string',
            'local_address' => 'required|string',

            'belongs_to_minority_group' => 'nullable|boolean',
            'is_disability_ph' => 'nullable|boolean',
            'is_transport_required' => 'nullable|boolean',

            'profile_image' => 'nullable|image|max:2048',
            'dob_photo_copy' => 'nullable|mimes:jpg,jpeg,png,pdf|max:2048',

            // ================= PARENT =================
            'father_name' => 'required|string|max:255',
            'mother_name' => 'required|string|max:255',

            'father_qualification' => 'nullable|string|max:255',
            'mother_qualification' => 'nullable|string|max:255',

            'father_occupation' => 'required|string|max:255',
            'mother_occupation' => 'required|string|max:255',

            'father_phone' => 'required|digits:10',
            'mother_phone' => 'required|digits:10',

            'father_voter_id' => 'nullable|string|max:50',
            'mother_voter_id' => 'nullable|string|max:50',

            'guardian_name' => 'nullable|string|max:255',
            'guardian_phone' => 'nullable|digits:10',
            'guardian_address' => 'nullable|string',

            // ================= CLASS =================
            'class_id' => 'required|integer',
            'section_id' => 'required|integer',
            'session_id' => 'required|integer',
            'financial_year_id' => 'required|integer',
        ]);


        DB::transaction(function () use ($request, $validated, $student) {

            // ================= FILES =================
            if ($request->hasFile('profile_image')) {
                $student->profile_image =
                    $request->file('profile_image')->store('students/profile', 'public');
            }

            if ($request->hasFile('dob_photo_copy')) {
                $student->dob_photo_copy =
                    $request->file('dob_photo_copy')->store('students/dob', 'public');
            }

            // ================= PARENT =================
            $student->parent->update([
                'father_name' => $validated['father_name'],
                'mother_name' => $validated['mother_name'],
                'father_qualification' => $validated['father_qualification'],
                'mother_qualification' => $validated['mother_qualification'],
                'father_occupation' => $validated['father_occupation'],
                'mother_occupation' => $validated['mother_occupation'],
                'father_phone' => $validated['father_phone'],
                'mother_phone' => $validated['mother_phone'],
                //'father_voter_id' => $validated['father_voter_id'],
                //'mother_voter_id' => $validated['mother_voter_id'],
                'guardian_name' => $validated['guardian_name'],
                'guardian_phone' => $validated['guardian_phone'],
                'guardian_address' => $validated['guardian_address'],
            ]);

            // ================= STUDENT =================
            $student->update([
                'name' => $validated['name'],
                'dob' => $validated['dob'],
                'gender' => $validated['gender'],
                'nationality' => $validated['nationality'],
                'mother_tongue' => $validated['mother_tongue'],
                'category' => $validated['category'],
                'cast' => $validated['cast'],
                'address' => $validated['address'],
                'local_address' => $validated['local_address'],
                'belongs_to_minority_group' => $request->boolean('belongs_to_minority_group'),
                'is_disability_ph' => $request->boolean('is_disability_ph'),
                'is_transport_required' => $request->boolean('is_transport_required'),
            ]);

            // ================= CLASS DETAILS =================
            $student->classDetails()
                ->where('status', 1)
                ->update([
                    'class_id' => $validated['class_id'],
                    'section_id' => $validated['section_id'],
                    'session_id' => $validated['session_id'],
                    'financial_year_id' => $validated['financial_year_id'],
                ]);
        });

        return redirect()
            ->route('students.index')
            ->with('success', 'Student updated successfully');
    }

    public function show($id)
    {
        $student = Student::with([
            'parent',
            'classDetails' => function ($q) {
                $q->where('status', 1);
            },
            'classDetails.class',
            'classDetails.section',
            'classDetails.financialYear',
        ])->findOrFail($id);

        $currentClass = $student->classDetails->first();

        return view('students.show', compact(
            'student',
            'currentClass'
        ));
    }




}
