<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Carbon\Carbon;
use DB;

use App\Models\SchoolClass;
use App\Models\SchoolSection;
use App\Models\Student;
use App\Models\AcademicYear;
use App\Models\ClassFeeConfig;
use App\Models\StudentMonthlyFee;

class StudentFeeController extends Controller
{
    public function index(Request $request)
    {
        /* -------------------------------------------------
            Filters
        ------------------------------------------------- */
        $financialYearId = $request->financial_year_id;
        $classId         = $request->class_id;
        $sectionId       = $request->section_id;
        $month           = $request->filled('month')
                            ? $request->month
                            : now()->format('Y-m');
        $search          = $request->q;

        /* -------------------------------------------------
            Masters
        ------------------------------------------------- */
        $academic_years = AcademicYear::where('status',1)->get();
        $active_academic_year = AcademicYear::where('status',1)->first();
        if(!$financialYearId){
            $financialYearId = $active_academic_year->id;
        }

        $classes = SchoolClass::where('status',1)->get();

        $sections = collect();
        if ($classId && $classId !== 'all') {
            $sections = SchoolSection::where('class_id', $classId)->get();
        }

        /* -------------------------------------------------
            Students Query
        ------------------------------------------------- */
        $students = Student::with([
            'activeClass.class',
            'activeClass.section'
        ])
        ->where('status', 1)

        // Filter by class
        ->when($classId && $classId !== 'all', function ($q) use ($classId) {
            $q->whereHas('activeClass', function ($qq) use ($classId) {
                $qq->where('class_id', $classId);
            });
        })

        // Filter by section
        ->when($sectionId, function ($q) use ($sectionId) {
            $q->whereHas('activeClass', function ($qq) use ($sectionId) {
                $qq->where('section_id', $sectionId);
            });
        })

        // Search
        ->when($search, function ($q) use ($search) {
            $q->where(function ($qq) use ($search) {
                $qq->where('name', 'like', "%$search%")
                ->orWhere('admission_no', 'like', "%$search%");
            });
        })

        ->paginate(25)
        ->withQueryString();


        
        /* -------------------------------------------------
            Fee Calculation (per student)
        ------------------------------------------------- */


        foreach ($students as $student) {

            $classId = optional($student->activeClass)->class_id;

            if (!$classId) {
                $student->total_fee  = 0;
                $student->fee_status = 'N/A';
                continue;
            }

            $configs = ClassFeeConfig::where('class_id', $classId)
                ->where('financial_year_id', $financialYearId)
                ->where('status', 1)
                ->get();

            $total = 0;
            $hasUnpaid = false;

            foreach ($configs as $cfg) {

                /* ===============================
                    RECURRING FEES
                =============================== */
                if ($cfg->fee_type === 'recurring') {

                    $paid = StudentMonthlyFee::where([
                        'student_id'        => $student->id,
                        'fee_head_id'       => $cfg->fee_head_id,
                        'financial_year_id' => $financialYearId,
                        'fee_for_month'     => $month
                    ])->exists();

                    if ($paid) {
                        continue;
                    }

                    $lateFee = 0;

                    if (
                        $cfg->due_day_of_month &&
                        now()->day > $cfg->due_day_of_month
                    ) {
                        $lateFee = $cfg->late_fee_amount ?? 0;
                    }

                    $total += ($cfg->amount + $lateFee);
                    $hasUnpaid = true;
                }

                /* ===============================
                    ONE-TIME FEES
                =============================== */
                if (
                    $cfg->fee_type === 'one_time' &&
                    $cfg->due_date &&
                    Carbon::parse($cfg->due_date)->format('Y-m') === $month
                ) {
                    $paid = StudentMonthlyFee::where([
                        'student_id'        => $student->id,
                        'fee_head_id'       => $cfg->fee_head_id,
                        'financial_year_id' => $financialYearId,
                    ])->exists();

                    if ($paid) {
                        continue;
                    }

                    $lateFee = 0;
                    if (now()->gt(Carbon::parse($cfg->due_date))) {
                        $lateFee = $cfg->late_fee_amount ?? 0;
                    }

                    $total += ($cfg->amount + $lateFee);
                    $hasUnpaid = true;
                }
            }

            $student->total_fee  = $total;
            $student->fee_status = $hasUnpaid ? 'unpaid' : 'paid';
        }



        // echo '<pre>';
        // echo $financialYearId;
        // print_r($students->toArray()); die;


        return view('student_fees.index', compact(
            'academic_years',
            'active_academic_year',
            'classes',
            'sections',
            'students'
        ));
    }



    /* =========================================================
        FETCH STUDENT FEES (AJAX)
    ========================================================= */
    public function getStudentFees(Student $student, Request $request)
    {
        $yearId = $request->financial_year_id;

        $classId = optional($student->activeClass)->class_id;
        if (!$classId) {
            return response()->json([]);
        }

        $academicYear = AcademicYear::findOrFail($yearId);

        /* ---------------------------------------
            CLASS + YEAR + ACTIVE FEES ONLY
        --------------------------------------- */
        $configs = ClassFeeConfig::with('feeHead')
            ->where('class_id', $classId)                 // ✅ FIX 1
            ->where('financial_year_id', $yearId)
            ->where('status', 1)
            ->get();

        $months = [];

        /* ---------------------------------------
            BUILD MONTH RANGE (AY → CURRENT)
        --------------------------------------- */
        $start = Carbon::parse($academicYear->year_from)->startOfMonth();
        $end   = now()->startOfMonth();

        while ($start <= $end) {

            $monthKey   = $start->format('Y-m');
            $monthLabel = $start->format('F Y');

            $months[$monthKey] = [
                'month'      => $monthKey,
                'label'      => $monthLabel,
                'fees'       => [],
                'total'      => 0,
                'selectable' => false
            ];

            foreach ($configs as $cfg) {

                /* ===============================
                    RECURRING FEES (MONTH WISE)
                =============================== */
                if ($cfg->fee_type === 'recurring') {

                    $paid = StudentMonthlyFee::where([
                        'student_id'        => $student->id,
                        'fee_head_id'       => $cfg->fee_head_id,
                        'financial_year_id'=> $yearId,
                        'fee_for_month'     => $monthKey
                    ])->exists();

                    if (!$paid) {

                        $lateFee = 0;

                        if ($cfg->due_day_of_month) {
                            $dueDate = Carbon::createFromFormat(
                                'Y-m-d',
                                $monthKey . '-' . str_pad($cfg->due_day_of_month, 2, '0', STR_PAD_LEFT)
                            );

                            if (now()->gt($dueDate)) {
                                $lateFee = $cfg->late_fee_amount ?? 0;
                            }
                        }

                        $months[$monthKey]['fees'][] = [
                            'fee_head_id' => $cfg->fee_head_id,
                            'name'        => $cfg->feeHead->name,
                            'type'        => 'recurring',
                            'amount'      => $cfg->amount,
                            'late_fee'    => $lateFee,
                            'total'       => $cfg->amount + $lateFee
                        ];

                        $months[$monthKey]['total'] += ($cfg->amount + $lateFee);
                        $months[$monthKey]['selectable'] = true;
                    }
                }

                /* ===============================
                    ONE-TIME FEES (DUE MONTH ONLY)
                =============================== */
                if ($cfg->fee_type === 'one_time' && $cfg->due_date) {

                    if (Carbon::parse($cfg->due_date)->format('Y-m') !== $monthKey) {
                        continue;
                    }

                    $alreadyPaid = StudentMonthlyFee::where([
                        'student_id'        => $student->id,
                        'fee_head_id'       => $cfg->fee_head_id,
                        'financial_year_id'=> $yearId
                    ])->exists();

                    if (!$alreadyPaid) {

                        $lateFee = now()->gt(Carbon::parse($cfg->due_date))
                            ? ($cfg->late_fee_amount ?? 0)
                            : 0;

                        $months[$monthKey]['fees'][] = [
                            'fee_head_id' => $cfg->fee_head_id,
                            'name'        => $cfg->feeHead->name,
                            'type'        => 'one_time',
                            'amount'      => $cfg->amount,
                            'late_fee'    => $lateFee,
                            'total'       => $cfg->amount + $lateFee
                        ];

                        $months[$monthKey]['total'] += ($cfg->amount + $lateFee);
                        $months[$monthKey]['selectable'] = true;
                    }
                }
            }

            $start->addMonth();
        }

        /* ---------------------------------------
            REMOVE EMPTY MONTHS
        --------------------------------------- */
        $months = array_values(array_filter($months, fn ($m) => count($m['fees']) > 0));

        return response()->json($months);
    }



    /* =========================================================
        STORE PAYMENT
    ========================================================= */
    public function storePayment(Request $request)
    {
        // $request->validate([
        //     'student_id'         => 'required|exists:students,id',
        //     'financial_year_id'  => 'required|exists:academic_years,id',
        //     'payment_method'     => 'required|string',
        //     'transaction_id'     => 'nullable|string',
        //     'months'             => 'required|array',
        //     'fees'               => 'required|array',
        // ]);

        // echo '<pre>';
        // print_r($request->toArray());die;

        DB::beginTransaction();

        try {
            

            foreach ($request->months as $month) {

                if (!isset($request->fees[$month])) {
                    continue;
                }

                foreach ($request->fees[$month] as $fee) {

                    // 🔒 Prevent duplicate charge
                    $alreadyPaid = StudentMonthlyFee::where([
                        'student_id'         => $request->student_id,
                        'fee_head_id'        => $fee['fee_head_id'],
                        'financial_year_id' => $request->financial_year_id,
                        'fee_for_month'     => $month,
                    ])->exists();

                    if ($alreadyPaid) {
                        continue;
                    }

                    StudentMonthlyFee::create([
                        'student_id'         => $request->student_id,
                        'fee_head_id'        => $fee['fee_head_id'],
                        'financial_year_id' => $request->financial_year_id,
                        'fee_for_month'     => $month,
                        'amount'            => $fee['amount'],
                        'late_fee'          => $fee['late_fee'] ?? 0,
                        'paid_date'         => now(),
                        'payment_method'    => $request->payment_method,
                        'transaction_id'    => $request->transaction_id,
                    ]);
                }
            }

            DB::commit();

            // return response()->json([
            //     'success' => true,
            //     'message' => 'Fee collected successfully'
            // ]);

            return back()->with('success', 'Fee collected successfully.');

        } catch (\Throwable $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to collect fee',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function monthDetails(Student $student, Request $request)
    {
        $month = $request->month;

        $student->load([
            'activeClass.class',
            'activeClass.section'
        ]);

        $fees = StudentMonthlyFee::with('feeHead')
            ->where('student_id', $student->id)
            ->where('fee_for_month', $month)
            ->get();

        return response()->json([
            'student' => [
                'name'          => $student->name,
                'admission_no'  => $student->admission_no,
                'class'         => optional($student->activeClass->class)->class_name,
                'section'       => optional($student->activeClass->section)->section_name,
                'roll_no'       => optional($student->activeClass)->roll_no,
            ],
            'month' => $month,
            'fees'  => $fees
        ]);
    }


    public function monthPendingFees(Student $student, Request $request)
    {
        $month = $request->month;
        $yearId = $request->financial_year_id;

        $classId = optional($student->activeClass)->class_id;
        if (!$classId) return response()->json([]);

        $configs = ClassFeeConfig::with('feeHead')
            ->where('class_id', $classId)
            ->where('financial_year_id', $yearId)
            ->where('status', 1)
            ->get();

        $fees = [];

        foreach ($configs as $cfg) {

            $alreadyPaid = StudentMonthlyFee::where([
                'student_id' => $student->id,
                'fee_head_id'=> $cfg->fee_head_id,
                'financial_year_id'=> $yearId,
                'fee_for_month'=> $month
            ])->exists();

            if ($alreadyPaid) continue;

            $late = 0;

            if ($cfg->fee_type === 'recurring' && $cfg->due_day_of_month) {
                if (now()->day > $cfg->due_day_of_month) {
                    $late = $cfg->late_fee_amount ?? 0;
                }
            }

            if ($cfg->fee_type === 'one_time' &&
                $cfg->due_date &&
                Carbon::parse($cfg->due_date)->format('Y-m') === $month
            )  {
                if (now()->gt($cfg->due_date)) {
                    $late = $cfg->late_fee_amount ?? 0;
                }
            }

            $fees[] = [
                'fee_head_id' => $cfg->fee_head_id,
                'name' => $cfg->feeHead->name,
                'amount' => $cfg->amount,
                'late_fee' => $late,
                'total' => $cfg->amount + $late
            ];
        }

        return response()->json([
            'month' => $month,
            'fees' => $fees
        ]);
    }

    public function paySingleMonth(Request $request)
    {
        $request->validate([
            'student_id'        => 'required|exists:students,id',
            'financial_year_id' => 'required|exists:financial_years,id',
            'fee_for_month'     => 'required',
            'payment_method'    => 'required',
            'transaction_id'    => 'required',
            'fees'              => 'required|array|min:1',
        ]);

        DB::transaction(function () use ($request) {

            foreach ($request->fees as $fee) {

                // 🛑 Prevent duplicate payment
                $exists = StudentMonthlyFee::where([
                    'student_id'        => $request->student_id,
                    'fee_head_id'       => $fee['fee_head_id'],
                    'financial_year_id' => $request->financial_year_id,
                    'fee_for_month'     => $request->fee_for_month
                ])->exists();

                if ($exists) {
                    continue;
                }

                StudentMonthlyFee::create([
                    'student_id'        => $request->student_id,
                    'fee_head_id'       => $fee['fee_head_id'],
                    'financial_year_id' => $request->financial_year_id,
                    'fee_for_month'     => $request->fee_for_month,

                    'amount'            => $fee['amount'],
                    'late_fee'          => $fee['late_fee'] ?? 0,

                    'paid_date'         => now(),
                    'payment_method'    => $request->payment_method,
                    'transaction_id'    => $request->transaction_id,
                ]);
            }
        });

        return back()->with('success', 'Fee collected successfully.');
    }

}
