<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;


use App\Models\SchoolClass;
use App\Models\SchoolSection;

use App\Models\StudentClassDetail;
use App\Models\ClassFeeConfig;
use Illuminate\Http\JsonResponse;

class ClassSetupController extends Controller
{
    public function index()
    {
        $classes = SchoolClass::with(['sections', 'students'])->get();

        return view('class.index', compact('classes'));
    }

    public function addClass(Request $request)
    {
        $request->validate(
            [
                'class_name' => [
                    'required',
                    'string',
                    'max:50',
                    'unique:classes,class_name',
                    'regex:/^[A-Za-z0-9 ]+$/',
                ],
            ],
            [
                'class_name.regex' => 'Class name can contain only letters, numbers, and spaces.',
                'class_name.unique' => 'This class already exists.',
            ]
        );

        SchoolClass::create([
            'class_name' => trim($request->class_name),
        ]);

        return back()->with('success', 'Class added successfully.');
    }



    public function updateClass(Request $request, $id)
    {
        $class = SchoolClass::findOrFail($id);

        $request->validate(
            [
                'class_name' => [
                    'required',
                    'string',
                    'max:50',
                    'regex:/^[A-Za-z0-9 ]+$/',
                    Rule::unique('classes', 'class_name')->ignore($class->id),
                ],
            ],
            [
                'class_name.regex'  => 'Class name can contain only letters, numbers, and spaces.',
                'class_name.unique' => 'This class already exists.',
            ]
        );

        $class->update([
            'class_name' => ucwords(trim($request->class_name)),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Class updated successfully.',
        ]);
    }



    // public function deleteClass($id)
    // {
    //     SchoolClass::findOrFail($id)->delete();

    //     return response()->json(['success' => true]);
    // }



    public function deleteClass($id): JsonResponse
    {
        $class = SchoolClass::findOrFail($id);

        /* --------------------------------------------
            Check students assigned to this class
        -------------------------------------------- */
        $hasStudents = StudentClassDetail::where('class_id', $id)
            ->where('status', 1) // active students only
            ->exists();

        if ($hasStudents) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete class. Students are already assigned to this class.'
            ], 422);
        }

        /* --------------------------------------------
            Check fee configuration exists
        -------------------------------------------- */
        $hasFeeConfig = ClassFeeConfig::where('class_id', $id)->exists();

        if ($hasFeeConfig) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete class. Fee configuration exists for this class.'
            ], 422);
        }

        /* --------------------------------------------
            Safe to delete
        -------------------------------------------- */
        $class->delete();

        return response()->json([
            'success' => true,
            'message' => 'Class deleted successfully.'
        ]);
    }



    public function addSection(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'class_id' => 'required|exists:classes,id',
                'section_name' => [
                    'required',
                    'string',
                    'size:1',
                    'regex:/^[A-Z]$/'
                ],
            ],
            [
                'section_name.regex' => 'Section must be a single letter (A–Z).'
            ]
        );

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        $sectionName = strtoupper(trim($request->section_name)); // A, B, C, A1

        // 🔒 Class-wise duplicate check
        $exists = SchoolSection::where('class_id', $request->class_id)
            ->where('section_name', $sectionName)
            ->exists();

        if ($exists) {
            return response()->json([
                'errors' => [
                    'section_name' => ['This section already exists for the selected class.']
                ]
            ], 422);
        }

        $section = SchoolSection::create([
            'class_id'     => $request->class_id,
            'section_name' => $sectionName,
            'status'       => 1
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Section added successfully.',
            'data'    => $section
        ]);
    }





    public function deleteSection($id): JsonResponse
    {
        $section = SchoolSection::findOrFail($id);

        /* --------------------------------------------
            Check students assigned to this section
        -------------------------------------------- */
        $hasStudents = StudentClassDetail::where('section_id', $id) 
            ->exists();

        if ($hasStudents) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete section. Students are assigned to this section.'
            ], 422);
        }

        /* --------------------------------------------
            Safe to delete
        -------------------------------------------- */
        $section->delete();

        return response()->json([
            'success' => true,
            'message' => 'Section deleted successfully.'
        ]);
    }

}
