<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

use App\Models\ExpenseHead;
use App\Models\IncomeHead;
use App\Models\TransactionIncome;
use App\Models\TransactionExpense;


class TransactionConfigController extends Controller
{
    public function index(Request $request)
    {
        $tab = $request->get('tab', 'income_head');

        $incomeHeads = IncomeHead::orderBy('id', 'asc')
            ->paginate(25, ['*'], 'income_head_page')
            ->withQueryString();

        $expenseHeads = ExpenseHead::orderBy('id', 'asc')
            ->paginate(25, ['*'], 'expense_head_page')
            ->withQueryString();

        return view('transaction_config.index', compact(
            'tab',
            'incomeHeads',
            'expenseHeads',
        ));
    }

    /* =========================
        Income HEAD
    ========================= */

    public function storeIncomeHead(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $exists = IncomeHead::where('name', $request->name)->exists();
        if ($exists) {
            return redirect()->route('transaction_config.index', [
                'tab' => 'income_head'
            ])->with('error', 'Income Head already exists!');
        }

        IncomeHead::create([
            'name' => $request->name,
            'description' => $request->description,
            'status' => 1
        ]);

        return redirect()->route('transaction_config.index', [
            'tab' => 'income_head'
        ])->with('success', 'Income Head Added!');
    }

    public function updateIncomeHead(Request $request, $id)
    {
        $incomeHead = IncomeHead::findOrFail($id);

        /*
        |--------------------------------------------------------------------------
        | STATUS TOGGLE (AJAX)
        |--------------------------------------------------------------------------
        */
        if ($request->has('status')) {


            try {

                // Update fee head status
                $incomeHead->update([
                    'status' => $request->boolean('status')
                ]);
                return response()->json([
                    'success' => true,
                    'message' => 'Status updated successfully'
                ]);

            } catch (\Throwable $e) {


                return response()->json([
                    'success' => false,
                    'message' => 'Unable to update income head status'
                ], 500);
            }
        }


        /*
        |--------------------------------------------------------------------------
        | EDIT (AJAX)
        |--------------------------------------------------------------------------
        */
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        $exists = IncomeHead::where('name', $request->name)
            ->where('id', '!=', $id)
            ->exists();

        if ($exists) {
            return response()->json([
                'errors' => [
                    'name' => ['Income Head already exists']
                ]
            ], 422);
        }

        $incomeHead->update([
            'name' => $request->name,
            'description' => $request->description
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Income Head updated successfully'
        ]);
    }

    public function deleteIncomeHead($id)
    {
        try {
            $incomeHead = IncomeHead::findOrFail($id);

            /* -----------------------------------------
            Restriction 2: Already collected from students
            ----------------------------------------- */
            $usedInTransaction = TransactionIncome::where('income_head_id', $id)->exists();

            if ($usedInTransaction) {
                return response()->json([
                    'success' => false,
                    'message' => 'Income Head already exists in transactions. Deletion not allowed.'
                ], 422);
            }

            $incomeHead->delete();

            return response()->json([
                'success' => true,
                'message' => 'Income Head deleted successfully'
            ]);

        } catch (\Throwable $e) {

            return response()->json([
                'success' => false,
                'message' => 'Unable to delete income head'
            ], 500);
        }
    }


    /* =========================
        Expense HEAD
    ========================= */

    public function storeExpenseHead(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $exists = ExpenseHead::where('name', $request->name)->exists();
        if ($exists) {
            return redirect()->route('transaction_config.index', [
                'tab' => 'expense_head'
            ])->with('error', 'Expense Head already exists!');
        }

        ExpenseHead::create([
            'name' => $request->name,
            'description' => $request->description,
            'status' => 1
        ]);

        return redirect()->route('transaction_config.index', [
            'tab' => 'expense_head'
        ])->with('success', 'Expense Head Added!');
    }

    public function updateExpenseHead(Request $request, $id)
    {
        $expenseHead = ExpenseHead::findOrFail($id);

        /*
        |--------------------------------------------------------------------------
        | STATUS TOGGLE (AJAX)
        |--------------------------------------------------------------------------
        */
        if ($request->has('status')) {

            try {

                $expenseHead->status = $request->boolean('status');
                $expenseHead->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Status updated successfully'
                ]);

            } catch (\Throwable $e) {


                return response()->json([
                    'success' => false,
                    'message' => 'Unable to update expense head status'
                ], 500);
            }
        }

        /*
        |--------------------------------------------------------------------------
        | EDIT (AJAX)
        |--------------------------------------------------------------------------
        */
        $validator = Validator::make($request->all(), [
            'name'        => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        $exists = ExpenseHead::where('name', $request->name)
            ->where('id', '!=', $id)
            ->exists();

        if ($exists) {
            return response()->json([
                'errors' => [
                    'name' => ['Expense Head already exists']
                ]
            ], 422);
        }

        $expenseHead->update([
            'name'        => $request->name,
            'description' => $request->description
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Expense Head updated successfully'
        ]);
    }


    public function deleteExpenseHead($id)
    {
        try {
            $expenseHead = ExpenseHead::findOrFail($id);

            /* -----------------------------------------
            Restriction 2: Already collected from students
            ----------------------------------------- */
            $usedInTransaction = TransactionExpense::where('expense_head_id', $id)->exists();

            if ($usedInTransaction) {
                return response()->json([
                    'success' => false,
                    'message' => 'Expense Head already exists in transactions. Deletion not allowed.'
                ], 422);
            }

            $expenseHead->delete();

            return response()->json([
                'success' => true,
                'message' => 'Expense Head deleted successfully'
            ]);

        } catch (\Throwable $e) {

            return response()->json([
                'success' => false,
                'message' => 'Unable to delete expense head'
            ], 500);
        }
    }



}
