<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

use App\Models\Vendor;
use App\Models\TransactionExpense;

class VendorController extends Controller
{
    public function index(Request $request)
    {

        $vendors = Vendor::orderBy('id', 'desc')
            ->paginate(25)
            ->withQueryString();

        return view('vendor.index', compact(
            'vendors'
        ));
    }

    /* =========================
        FEE HEAD
    ========================= */

    public function storeVendor(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $exists = Vendor::where('name', $request->name)->exists();
        if ($exists) {
            return redirect()->route('vendors.index')->with('error', $request->name.' Vendor already exists!');
        }

        Vendor::create([
            'name' => $request->name,
            'description' => $request->description,
            'status' => 1
        ]);

        return redirect()->route('vendors.index')->with('success', 'Vendor Added!');
    }

    /**
     * 🔥 SINGLE FUNCTION
     * Edit + Status Toggle (AJAX)
     */
    public function updateVendor(Request $request, $id)
    {
        $vendor = Vendor::findOrFail($id);

        /*
        |--------------------------------------------------------------------------
        | STATUS TOGGLE (AJAX)
        |--------------------------------------------------------------------------
        */
        if ($request->has('status')) {

            //DB::beginTransaction();

            try {

                // Update fee head status
                $vendor->update([
                    'status' => $request->status ? 1 : 0
                ]);


                return response()->json([
                    'success' => true,
                    'message' => $request->status
                        ? 'Vendor activated successfully'
                        : 'Vendor deactivated successfully'                
                ]);

            } catch (\Throwable $e) {

                //DB::rollBack();

                return response()->json([
                    'success' => false,
                    'message' => 'Unable to update vendor status'
                ], 500);
            }
        }


        /*
        |--------------------------------------------------------------------------
        | EDIT (AJAX)
        |--------------------------------------------------------------------------
        */
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        $exists = Vendor::where('name', $request->name)
            ->where('id', '!=', $id)
            ->exists();

        if ($exists) {
            return response()->json([
                'errors' => [
                    'name' => ['Vendor already exists']
                ]
            ], 422);
        }

        $vendor->update([
            'name' => $request->name,
            'description' => $request->description
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Vendor updated successfully'
        ]);
    }

    public function deleteVendor($id)
    {
        try {
            $vendor = Vendor::findOrFail($id);

            /* -----------------------------------------
            Restriction 1: Used in Transaction
            ----------------------------------------- */
            $usedInTransaction = TransactionExpense::where('source_type','vendor')->where('source_id', $id)->exists();

            if ($usedInTransaction) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vendor is already used in transaction'
                ], 422);
            }

            $vendor->delete();

            return response()->json([
                'success' => true,
                'message' => 'Vendor deleted successfully'
            ]);

        } catch (\Throwable $e) {

            return response()->json([
                'success' => false,
                'message' => 'Unable to delete vendor'
            ], 500);
        }
    }


    public function search(Request $request)
    {
        $q = $request->q;

        return Vendor::where('name', 'like', "%{$q}%")
            ->where('status',1)
            ->limit(20)
            ->get(['id','name']);
    }



}
