@extends('layouts.app')

@section('title','Vendors')

@section('breadcrumb')
<li class="breadcrumb-item active">Vendors</li>
@endsection

@section('content')
<div id="vendor-page" class="page-content">
    <h1 class="mb-4">Vendors</h1>
    <div class="card mb-4">
        <div class="card-header bg-light">
            <h5 class="mb-0">Add New Vendor</h5>
        </div>
        <div class="card-body">
            <form method="POST" action="{{ route('vendors.storeVendor') }}">
                @csrf
                <div class="row">
                    <div class="col-md-5">
                        <input type="text" name="name" class="form-control" placeholder="Vendor Name" required>
                    </div>
                    <div class="col-md-5">
                        <input type="text" name="description" class="form-control" placeholder="Description" required>
                    </div>
                    <div class="col-md-2">
                        <button class="btn btn-primary w-100">Add</button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <div class="card">
        <div class="card-header bg-light">
            <h5 class="mb-0">Vendor List</h5>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-sm">
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Description</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($vendors as $vendor)
                        <tr>
                            <td>{{ $vendor->name }}</td>
                            <td title="{{$vendor->description}}">{{ \Illuminate\Support\Str::words($vendor->description ?? '', 10, '...') }}</td>

                            <td>
                                <div class="form-check form-switch">
                                    <input
                                        class="form-check-input vendor-status"
                                        type="checkbox"
                                        id="status_{{ $vendor->id }}"
                                        data-id="{{ $vendor->id }}"
                                        data-url="{{ route('vendors.updateVendor', $vendor->id) }}"
                                        {{ $vendor->status ? 'checked' : '' }}
                                    >
                                </div>
                            </td>
                            <td>
                                <button
                                    type="button"
                                    class="btn btn-sm btn-outline-primary edit-vendor"
                                    data-id="{{ $vendor->id }}"
                                    data-name="{{ $vendor->name }}"
                                    data-description="{{ $vendor->description }}"
                                >
                                    Edit
                                </button>

                                <button
                                    type="button"
                                    class="btn btn-sm btn-outline-danger deleteVendorBtn"
                                    data-url="{{ route('vendors.deleteVendor', $vendor->id) }}"
                                >
                                    Delete
                                </button>


                            </td>
                        </tr>
                        @empty
                            <tr>
                                <td colspan="8" class="text-center text-muted">
                                    No vendor found!
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
            <div class="d-flex justify-content-end mt-3">
                {{ $vendors->links() }}
            </div>
        </div>
    </div>
</div>
@endsection

@section('modal')

    <div class="modal fade" id="editVendorModal" tabindex="-1">
        <div class="modal-dialog">
            <form id="editVendorForm">
                @csrf
                @method('PUT')

                <input type="hidden" id="edit_id">

                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Edit Vendor</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>

                    <div class="modal-body">

                        <div class="mb-3">
                            <label class="form-label">Vendor Name *</label>
                            <input type="text" class="form-control" id="edit_name" name="name">
                            <small class="text-danger d-none" id="error_name"></small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" id="edit_description" name="description"></textarea>
                            <small class="text-danger d-none" id="error_description"></small>
                        </div>

                    </div>


                    <div class="modal-footer">
                        <button type="submit" class="btn btn-primary" id="saveBtn">
                            <span id="saveText">Update</span>
                            <span id="saveSpinner" class="spinner-border spinner-border-sm d-none"></span>
                        </button>
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    </div>

                </div>
            </form>
        </div>
    </div>



@endsection

@section('script')
    <script>
        $(document).ready(function() {
            $(document).on('change', '.vendor-status', function () {
                let checkbox = $(this);
                let vendor_status = checkbox.prop('checked') ? 1 : 0;

                $.ajax({
                    url: checkbox.data('url'),
                    type: "PUT",             
                    //dataType: "json",
                    data: {
                        _token: "{{ csrf_token() }}",    
                        status: vendor_status
                    },
                    success: function (data) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Success',
                            text: data.message
                        });
                    },
                    error: function (xhr) {
                        let message = 'Status not updated';

                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            message = xhr.responseJSON.message;
                        }

                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: message
                        });

                        checkbox.prop('checked', !checkbox.prop('checked'));
                    }
                });
            });

        });

    </script>
    <script>
        document.addEventListener('click', function (e) {
            if (!e.target.classList.contains('deleteVendorBtn')) return;

            const url = e.target.dataset.url;
            const row = e.target.closest('tr');

            Swal.fire({
                title: "Are you sure?",
                text: "This Vendor will be deleted!",
                icon: "warning",
                showCancelButton: true,
                confirmButtonColor: "#e3342f",
                cancelButtonColor: "#6c757d",
                confirmButtonText: "Yes, delete it",
                cancelButtonText: "Cancel",
            }).then(result => {
                if (!result.isConfirmed) return;

                fetch(url, {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                })
                .then(res => res.json())
                .then(res => {
                    if (res.success) {
                        row.remove();
                        Swal.fire('Deleted!', res.message, 'success');
                    } else {
                        Swal.fire('Error!', res.message, 'error');
                    }
                });
            });
        });

    </script>

    <script>
        document.addEventListener('DOMContentLoaded', function () {

            let editModal = new bootstrap.Modal(
                document.getElementById('editVendorModal')
            );

            const saveBtn = document.getElementById('saveBtn');
            const saveText = document.getElementById('saveText');
            const saveSpinner = document.getElementById('saveSpinner');

            function resetErrors() {
                document.querySelectorAll('[id^="error_"]').forEach(el => {
                    el.classList.add('d-none');
                    el.innerText = '';
                });
            }

            function toggleLoading(state) {
                saveBtn.disabled = state;
                saveSpinner.classList.toggle('d-none', !state);
                saveText.classList.toggle('d-none', state);
            }

            // Open modal
            document.querySelectorAll('.edit-vendor').forEach(btn => {
                btn.addEventListener('click', function () {

                    resetErrors();

                    document.getElementById('edit_id').value = this.dataset.id;
                    document.getElementById('edit_name').value = this.dataset.name;
                    document.getElementById('edit_description').value = this.dataset.description;

                    editModal.show();
                });
            });

            // AJAX Save
            document.getElementById('editVendorForm').addEventListener('submit', function (e) {
                e.preventDefault();

                resetErrors();
                toggleLoading(true);

                let id = document.getElementById('edit_id').value;

                fetch("{{ url('vendors/vendor') }}/" + id, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'X-HTTP-Method-Override': 'PUT',
                        'Accept': 'application/json'
                    },
                    body: new FormData(this)
                })
                .then(async res => {
                    toggleLoading(false);

                    if (!res.ok) {
                        let data = await res.json();
                        throw data;
                    }
                    return res.json();
                })
                .then(data => {

                    Swal.fire({
                        icon: 'success',
                        title: 'Updated',
                        text: data.message,
                        timer: 1500,
                        showConfirmButton: false
                    });

                    const newName = document.getElementById('edit_name').value;
                    const newDescription = document.getElementById('edit_description').value;

                    // Find edit button
                    let editBtn = document.querySelector(
                        `.edit-vendor[data-id="${id}"]`
                    );

                    // Update table row text
                    let row = editBtn.closest('tr');
                    row.children[0].innerText = newName;
                    row.children[1].innerText = newDescription;

                    //  Update data attributes (IMPORTANT)
                    editBtn.dataset.name = newName;
                    editBtn.dataset.description = newDescription;

                    editModal.hide();
                })

                .catch(err => {
                    toggleLoading(false);

                    if (err.errors) {
                        Object.keys(err.errors).forEach(field => {
                            let errorEl = document.getElementById('error_' + field);
                            if (errorEl) {
                                errorEl.innerText = err.errors[field][0];
                                errorEl.classList.remove('d-none');
                            }
                        });
                    } else if (err.message) {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: err.message
                        });
                    }
                });
            });

        });
    </script>
@endsection

